# youtube_search.tcl — Recherche YouTube v3 pour Eggdrop (debug avancé)
#
# Cmd publiques :  !yt <recherche>  ou  .yt <recherche>
# Cmd privée    :  /msg <bot> yt <recherche>
# Outils debug  :  ytdebug <0|1|2|3>  |  ythealth  |  ytver
# Activation    :  .chanset #canal +YoutubeSearch
# Dépendances   :  tcllib (json), http, tls (https)
# Debian        :  apt install -y tcllib tcl-tls
#
# =========================
#         VERSION
# =========================
# SemVer du script (modifier à chaque release)
# Historique succinct :
# - 1.2.1 (2025-10-21) : Tag [YouTube] avec "You" noir sur fond blanc, "Tube" blanc sur fond rouge.
#                        Séparateurs en orange. (aucune autre modif fonctionnelle)
# - 1.2.0 (2025-10-21) : ajout du versioning (yt::version, yt::build_date),
#                        UA dynamique, commandes ytver/msg ytver, logs enrichis.
# - 1.1.x (historique) : base fournie par l’utilisateur.
#
# Note : la version est aussi injectée dans l'User-Agent HTTP.

# --------- PREREQUIS (check silencieux) ----------
if {[catch {package require json} err]} {
    putlog "youtube_search.tcl: Tcl 'json' requis (Debian: apt install tcllib). Script désactivé."
    return
}
if {[catch {package require http} err]} {
    putlog "youtube_search.tcl: Tcl 'http' requis. Script désactivé."
    return
}
if {[catch {package require tls} err]} {
    putlog "youtube_search.tcl: Tcl 'tls' requis pour HTTPS (Debian: apt install tcl-tls). Script désactivé."
    return
}
::http::register https 443 ::tls::socket
# (UA sera réinitialisé après définition de yt::version)

# --------- NAMESPACE & CONFIG ----------
namespace eval yt {
    # --- Versionning ---
    variable version     "1.2.1"
    variable build_date  "2025-10-21"

    # --- Config runtime ---
    variable apikey
    variable cooldown 3       ;# anti-spam par canal (s)
    variable use_colors 1     ;# 1: tag mIRC [YouTube], 0: texte simple
    variable debug 3          ;# 0 muet, 1 normal, 2 détaillé, 3 très verbeux
    variable lastcall         ;# dict chan -> epoch (cooldown)
    set lastcall [dict create]

    # Mémoire debug (derniers appels HTTP)
    variable last_search_url
    variable last_search_code
    variable last_search_bodylen
    variable last_search_sample
    variable last_videos_url
    variable last_videos_code
    variable last_videos_bodylen
    variable last_videos_sample

    # Clé API : conf > ENV > placeholder
    if {![info exists apikey]} {
        if {[info exists ::env(YOUTUBE_APIKEY)]} {
            set apikey $::env(YOUTUBE_APIKEY)
        } else {
            set apikey "YOUR_API_KEY_HERE"
        }
    }
}

# Maintenant qu’on a la version, on met à jour l’User-Agent HTTP.
http::config -useragent "eggdrop-youtube/${yt::version}" -urlencoding utf-8

# Flag de chaîne
setudef flag YoutubeSearch

# Binds
bind pub  - "yt"       yt::cmd_pub
bind pubm - "*"        yt::cmd_pubm
bind msg  - "yt"       yt::cmd_msg
bind pub  n "ytdebug"  yt::cmd_setdebug
bind pub  n "ythealth" yt::cmd_health
bind msg  - "ytdebug"  yt::cmd_setdebug_msg
bind msg  - "ythealth" yt::cmd_health_msg
bind pub  - "ytver"    yt::cmd_version
bind msg  - "ytver"    yt::cmd_version_msg

# --------- LOG & MASQUAGE ----------
proc yt::dbg {lvl msg} {
    variable debug
    if {$debug >= $lvl} { putlog "yt: $msg" }
}
proc yt::mask_key {k} {
    if {$k eq ""} { return "<empty>" }
    set l [string length $k]
    if {$l <= 8} { return "***" }
    return "[string range $k 0 3]***[string range $k end-3 end]"
}
proc yt::mask_url {u} {
    if {[regexp {(key=)([^&]+)} $u -> pre val]} {
        set masked "[string range $val 0 3]***[string range $val end-3 end]"
        return [string map [list "$pre$val" "$pre$masked"] $u]
    }
    return $u
}

# --------- HELPERS ----------
proc yt::dict_getdef {d key def} {
    if {[dict exists $d $key]} { return [dict get $d $key] }
    return $def
}
proc yt::enabled {chan} {
    if {![validchan $chan]} { return 0 }
    return [channel get $chan YoutubeSearch]
}
proc yt::cooldown_ok {chan} {
    variable lastcall
    variable cooldown
    set now [clock seconds]
    if {![info exists lastcall]} { set lastcall [dict create] }
    set last [yt::dict_getdef $lastcall $chan 0]
    if {$now - $last < $cooldown} { return 0 }
    dict set lastcall $chan $now
    return 1
}
proc yt::thousands {n} {
    set s [string trim $n]; set out ""; set len [string length $s]; set c 0
    for {set i [expr {$len-1}]} {$i>=0} {incr i -1} {
        append out [string index $s $i]; incr c
        if {$c%3==0 && $i!=0} { append out "\u202f" }
    }
    return [string reverse $out]
}
proc yt::iso8601_to_hms {iso} {
    if {$iso eq ""} { return "" }
    set t [string trimleft $iso "PT"]; set h ""; set m ""; set s ""
    if {[regexp {(\d+)H} $t -> hh]} { set h "${hh}h " }
    if {[regexp {(\d+)M} $t -> mm]} { set m "${mm}mn " }
    if {[regexp {(\d+)S} $t -> ss]} { set s "${ss}s" }
    return [string trim "$h$m$s"]
}
proc yt::tag {} {
    variable use_colors
    if {$use_colors} {
        # \002=Bold \003=Color \017=Reset
        # "You" : texte noir (1) sur fond blanc (0)
        # "Tube": texte blanc (0) sur fond rouge (4)
        return "\002\[\0031,0You\0030,4Tube\003\002\]\017"
    }
    return "[YouTube]"
}

# HTTP helper (log verbeux + masquage clé)
proc yt::http_get {baseUrl argsDict which} {
    variable last_search_url
    variable last_search_code
    variable last_search_bodylen
    variable last_search_sample
    variable last_videos_url
    variable last_videos_code
    variable last_videos_bodylen
    variable last_videos_sample

    set url "${baseUrl}?[http::formatQuery {*}$argsDict]"
    set masked [yt::mask_url $url]
    yt::dbg 3 "$which GET $masked"

    set code 0
    set data ""
    set status ""

    if {[catch {
        set tok [http::geturl $url -timeout 10000 -headers {Accept application/json}]
        set code   [http::ncode  $tok]
        set data   [http::data   $tok]
        set status [http::status $tok]
        http::cleanup $tok
    } err]} {
        yt::dbg 1 "$which HTTP error: $err"
        set status "error"
    }

    set bodylen [string length $data]
    set sample  [string range $data 0 300]
    yt::dbg 2 "$which code=$code status=$status bodyLen=$bodylen"
    yt::dbg 3 "$which bodySample=[string map {\n \\n \r \\r \t \\t} $sample]"

    if {$which eq "search"} {
        set last_search_url      $masked
        set last_search_code     $code
        set last_search_bodylen  $bodylen
        set last_search_sample   $sample
    } else {
        set last_videos_url      $masked
        set last_videos_code     $code
        set last_videos_bodylen  $bodylen
        set last_videos_sample   $sample
    }
    return [list $code $data]
}

# --------- API YOUTUBE ----------
proc yt::api_search_videoid {q} {
    variable apikey
    if {$apikey eq "" || $apikey eq "REPLACE_WITH_YOUR_KEY"} {
        error "YOUTUBE_APIKEY non défini (mask=[yt::mask_key $apikey])"
    }
    set params [dict create \
        part snippet \
        q $q \
        type video \
        maxResults 1 \
        key $apikey \
        fields items/id/videoId]
    lassign [yt::http_get "https://www.googleapis.com/youtube/v3/search" $params "search"] code body
    if {$code != 200} {
        if {$code == 403} { yt::dbg 1 "search: 403 Forbidden (quota/key)" }
        return ""
    }
    if {[catch {set d [json::json2dict $body]} jerr]} {
        yt::dbg 1 "search: JSON parse error: $jerr"
        return ""
    }
    if {![dict exists $d items]} { return "" }
    set items [dict get $d items]
    if {[llength $items]==0} { return "" }
    set first [lindex $items 0]
    if {![dict exists $first id] || ![dict exists [dict get $first id] videoId]} { return "" }
    return [dict get $first id videoId]
}
proc yt::api_video_details {vid} {
    variable apikey
    set params [dict create \
        part "snippet,contentDetails,statistics" \
        id $vid \
        key $apikey \
        fields "items(snippet(title),contentDetails(duration),statistics(viewCount))"]
    lassign [yt::http_get "https://www.googleapis.com/youtube/v3/videos" $params "videos"] code body
    if {$code != 200} {
        if {$code == 403} { yt::dbg 1 "videos: 403 Forbidden (quota/key)" }
        return {}
    }
    if {[catch {set d [json::json2dict $body]} jerr]} {
        yt::dbg 1 "videos: JSON parse error: $jerr"
        return {}
    }
    if {![dict exists $d items]} { return {} }
    set items [dict get $d items]
    if {[llength $items]==0} { return {} }
    set it [lindex $items 0]

    set title ""; set dur ""; set views ""
    if {[dict exists $it snippet title]}           { set title [dict get $it snippet title] }
    if {[dict exists $it contentDetails duration]} { set dur   [dict get $it contentDetails duration] }
    if {[dict exists $it statistics viewCount]}    { set views [dict get $it statistics viewCount] }
    return [dict create title $title duration $dur views $views]
}

# --------- COEUR ----------
proc yt::do_search {nick chan query} {
    if {![yt::enabled $chan]} { yt::dbg 2 "skip: -YoutubeSearch sur $chan"; return }
    if {![yt::cooldown_ok $chan]} { yt::dbg 2 "cooldown $chan"; return }
    set query [string trim $query]
    if {$query eq ""} {
        puthelp "NOTICE $nick :Usage: yt <recherche>"
        return
    }
    variable apikey
    yt::dbg 1 "search '$query' (chan=$chan by $nick) key=[yt::mask_key $apikey]"
    if {[catch {set vid [yt::api_search_videoid $query]} err]} {
        puthelp "NOTICE $nick :YouTube API error: $err"
        return
    }
    if {$vid eq ""} {
        yt::dbg 1 "no videoId found (voir ythealth / logs search)"
        puthelp "NOTICE $nick :Aucun résultat pour: $query"
        return
    }
    set det [yt::api_video_details $vid]
    if {[dict size $det]==0} {
        yt::dbg 1 "empty details for vid=$vid (voir logs videos)"
        puthelp "NOTICE $nick :Impossible de récupérer les détails."
        return
    }
    set title [yt::dict_getdef $det title ""]
    set dur   [yt::iso8601_to_hms [yt::dict_getdef $det duration ""]]
    set views [yt::dict_getdef $det views ""]
    if {$views ne ""} { set views "vues [yt::thousands $views]" }

    set url "https://www.youtube.com/watch?v=$vid"
    set tag [yt::tag]

    # Séparateur orange : \0037 = orange ; \003 pour revenir aux couleurs par défaut
    set sep " \0037-\003 "

    # Assemblage (évite de recolorer un '-' dans le titre)
    set msg "$tag $url${sep}$title"
    if {$dur ne ""}   { append msg "${sep}$dur" }
    if {$views ne ""} { append msg "${sep}$views" }

    yt::dbg 2 "send: ($nick) $msg"
    putserv "PRIVMSG $chan :($nick) $msg"
}

# --------- HANDLERS ----------
proc yt::cmd_pub {nick uhost hand chan text} {
    yt::do_search $nick $chan [string trim $text]
}
proc yt::cmd_pubm {nick uhost hand chan text} {
    if {![regexp -nocase {^([!.])yt(?:\s+|$)(.*)} $text -> _ rest]} { return }
    yt::do_search $nick $chan [string trim $rest]
}
proc yt::cmd_msg {nick uhost hand text} {
    set query [string trim $text]
    if {$query eq ""} {
        puthelp "NOTICE $nick :Usage: yt <recherche>"
        return
    }
    set posted 0
    foreach c [channels] {
        if {[ischan $c] && [yt::enabled $c]} {
            yt::do_search $nick $c $query
            set posted 1
            break
        }
    }
    if {!$posted} {
        if {[catch {set vid [yt::api_search_videoid $query]} err]} {
            puthelp "NOTICE $nick :YouTube API error: $err"; return
        }
        if {$vid eq ""} { puthelp "NOTICE $nick :Aucun résultat pour: $query"; return }
        set det [yt::api_video_details $vid]
        if {[dict size $det]==0} { puthelp "NOTICE $nick :Impossible de récupérer les détails."; return }
        set title [yt::dict_getdef $det title ""]
        set dur   [yt::iso8601_to_hms [yt::dict_getdef $det duration ""]]
        set views [yt::dict_getdef $det views ""]
        if {$views ne ""} { set views "vues [yt::thousands $views]" }
        set url "https://www.youtube.com/watch?v=$vid"
        set txt "$url - $title"; if {$dur ne ""} { append txt " - $dur" }; if {$views ne ""} { append txt " - $views" }
        puthelp "NOTICE $nick :$txt"
    }
}

# --------- COMMANDES DEBUG / VERSION ----------
proc yt::cmd_setdebug {nick uhost hand chan text} {
    variable debug
    set v [string trim $text]
    if {![string is integer -strict $v]} { putserv "PRIVMSG $chan :$nick: usage: ytdebug 0|1|2|3"; return }
    set debug $v
    putserv "PRIVMSG $chan :$nick: debug=$debug"
}
proc yt::cmd_setdebug_msg {nick uhost hand text} {
    variable debug
    set v [string trim $text]
    if {![string is integer -strict $v]} { puthelp "NOTICE $nick :usage: ytdebug 0|1|2|3"; return }
    set debug $v
    puthelp "NOTICE $nick :debug=$debug"
}
proc yt::cmd_health {nick uhost hand chan text} {
    set res [yt::health_probe]
    putserv "PRIVMSG $chan :$nick: $res"
}
proc yt::cmd_health_msg {nick uhost hand text} {
    set res [yt::health_probe]
    puthelp "NOTICE $nick :$res"
}
proc yt::health_probe {} {
    set url "https://www.googleapis.com/robots.txt"
    set code 0; set status ""; set data ""
    if {[catch {
        set tok [http::geturl $url -timeout 8000]
        set code   [http::ncode $tok]
        set status [http::status $tok]
        set data   [http::data $tok]
        http::cleanup $tok
    } err]} {
        return "health: HTTP error: $err"
    }
    set blen [string length $data]
    set smp  [string range $data 0 120]
    return "health: code=$code status=$status bodyLen=$blen sample=[string map {\n \\n \r \\r \t \\t} $smp]"
}

# Version: texte détaillé + environnements libs
proc yt::version_string {} {
    set httpv  [package provide http]
    set tlsv   [package provide tls]
    set jsonv  [package provide json]
    set ua     [lindex [http::config] [lsearch -exact [http::config] -useragent]+1]
    set masked [yt::mask_key $::yt::apikey]
    return "youtube_search.tcl v$::yt::version (build $::yt::build_date) | http=$httpv tls=$tlsv json=$jsonv | UA='$ua' | key=$masked"
}
proc yt::cmd_version {nick uhost hand chan text} {
    putserv "PRIVMSG $chan :$nick: [yt::version_string]"
}
proc yt::cmd_version_msg {nick uhost hand text} {
    puthelp "NOTICE $nick :[yt::version_string]"
}

# --------- FIN CHARGEMENT ----------
set _mk [yt::mask_key $yt::apikey]
putlog "youtube_search.tcl chargé — v$yt::version (build $yt::build_date) — cmds: !yt/.yt — flags: +YoutubeSearch — debug=$yt::debug — key=$_mk"
