########################################################################################
# VDM (viedemerde.fr) pour Eggdrop — v1.1.2 (FIX CDATA IF EXPRESSION, 1 LINE, ID, ENDS VDM)
#
# Commande :
#   !vdm -> VDM aléatoire (1 seule ligne) avec ID, finit par "VDM"
#
# Chanset :
#   +VDM   (ex: .chanset #chan +VDM)
#
# Pré-requis :
#   Eggdrop >= 1.8.4 (build 1080404), Tcl >= 8.6, package http (tcllib)
#   TLS recommandé (tcl-tls)
#
# Debug :
#   console +d
#   .tcl set ::vdm::debug 2
########################################################################################

if {[info commands ::vdm::unload] eq "::vdm::unload"} { ::vdm::unload }

# Safety: si un script mal collé a injecté "^>" (vu chez toi), on neutralise.
if {[info commands ::^>] eq ""} { proc ::^> {args} { return } }

# Check Eggdrop build
set ::vdm__build [lindex [split $::version] 1]
if {![string is integer -strict $::vdm__build] || $::vdm__build < 1080404} {
  putloglev o * "\00304\[VDM - erreur\]\003 Eggdrop trop ancien: \00304${::version}\003 (>= 1.8.4 requis)."
  unset -nocomplain ::vdm__build
  return
}
unset -nocomplain ::vdm__build

if {[catch {package require Tcl 8.6}]} {
  putloglev o * "\00304\[VDM - erreur\]\003 Tcl 8.6+ requis (actuel: \00304${::tcl_version}\003)."
  return
}
if {[catch {package require http} herr]} {
  putloglev o * "\00304\[VDM - erreur\]\003 package 'http' requis (tcllib). Erreur: \00304$herr\003."
  return
}

namespace eval ::vdm {
  proc unload {args} {
    set ns [namespace current]
    putlog "VDM: nettoyage du namespace $ns..."
    if {[info exists ::vdm::proto_registered] && $::vdm::proto_registered} {
      catch { ::http::unregister httpsvdm }
      putlog "VDM: protocole httpsvdm unregister."
    }
    foreach b [binds *] {
      if {[llength $b] < 5} { continue }
      set cmd [lindex $b 4]
      if {[string match "${ns}::*" $cmd]} {
        unbind [lindex $b 0] [lindex $b 1] [lindex $b 2] $cmd
      }
    }
    namespace delete $ns
  }

  variable version "1.1.2"

  variable rss_url "https://www.viedemerde.fr/rss"
  variable timeout          15000
  variable max_redirects    5
  variable http_cooldown_ms 400
  variable last_http_ts     0

  variable max_attempts 4
  variable backoff_s    {0 1 2 4}

  variable debug 0
  variable max_len 350

  variable id_fg    1
  variable id_bg    15
  variable quote_fg 0
  variable quote_bg 14

  variable last_rid ""
  variable last_rid_ts 0
  variable avoid_same_for_s 120

  variable user_agent "Mozilla/5.0 (X11; Linux x86_64; rv:134.0) Gecko/20100101 Firefox/134.0"

  variable have_tls 0
  variable use_wrapper_https 1
  variable proto_registered 0
}

setudef flag VDM
expr {srand([clock clicks])}

# ---------------- logs (debug -> console +d) ----------------
proc ::vdm::log {lvl tag msg} {
  if {$::vdm::debug < $lvl} { return }
  putloglev d * "VDM($tag): $msg"
}
proc ::vdm::log_info  {msg} { ::vdm::log 1 INFO  $msg }
proc ::vdm::log_http  {msg} { ::vdm::log 2 HTTP  $msg }
proc ::vdm::log_parse {msg} { ::vdm::log 2 PARSE $msg }
proc ::vdm::log_sniff {msg} { ::vdm::log 2 SNIFF $msg }
proc ::vdm::log_err   {msg} { putloglev o * "VDM(ERR): $msg" }

# ---------------- TLS wrapper (comme DTC) ----------------
if {![catch {package require tls} terr]} {
  set ::vdm::have_tls 1
} else {
  set ::vdm::have_tls 0
  putlog "VDM: TLS indisponible ($terr)."
}

proc ::vdm::_tls_socket {args} {
  set async 0; set host ""; set port ""
  set i 0; set n [llength $args]
  while {$i < $n} {
    set a [lindex $args $i]
    if {[string match -* $a]} {
      if {$a eq "-async"} { set async 1 }
    } elseif {$host eq ""} {
      set host $a
    } elseif {$port eq ""} {
      set port $a
    }
    incr i
  }
  if {$host eq "" || $port eq ""} { error "_tls_socket: args invalides: $args" }
  set s [::socket $host $port]
  catch { fconfigure $s -translation binary -encoding binary -buffering none }
  if {$async} { catch { fconfigure $s -blocking 0 } }

  set ok 0
  foreach essai {
    {-autoservername 1 -servername __HOST__}
    {-servername __HOST__}
    {}
  } {
    set cmd [list ::tls::import $s]
    if {[llength $essai]} {
      foreach {k v} $essai {
        if {$v eq "__HOST__"} { set v $host }
        lappend cmd $k $v
      }
    }
    if {![catch { {*}$cmd }]} { set ok 1; break }
  }
  if {!$ok} { catch {close $s}; error "tls::import a échoué pour $host:$port" }
  return $s
}

if {$::vdm::have_tls} {
  if {![catch {::http::register httpsvdm 443 ::vdm::_tls_socket} regerr]} {
    set ::vdm::proto_registered 1
    ::vdm::log_info "protocole httpsvdm enregistré via http::register"
  } else {
    ::vdm::log_info "protocole httpsvdm non enregistré (déjà là ou erreur): $regerr"
    set ::vdm::proto_registered 0
  }
}

proc ::vdm::_rewrite_url {url} {
  if {$::vdm::have_tls && $::vdm::use_wrapper_https && [string match -nocase "https://*" $url]} {
    return [string map {"https://" "httpsvdm://"} $url]
  }
  return $url
}

catch { ::http::config -useragent $::vdm::user_agent }

# ---------------- helpers texte ----------------
proc ::vdm::html_unescape_min {s} {
  set s [string map [list "&amp;" "&" "&lt;" "<" "&gt;" ">" "&quot;" "\"" "&#039;" "'" "&nbsp;" " "] $s]
  regsub -all {\s+} $s " " s
  return [string trim $s]
}
proc ::vdm::strip_tags_one_line {html} {
  set s $html
  regsub -all -nocase {<br\s*/?>} $s " " s
  regsub -all {<[^>]+>} $s "" s
  return [::vdm::html_unescape_min $s]
}
proc ::vdm::truncate_one_line {s} {
  regsub -all {\r|\n} $s " " s
  regsub -all {\s+} $s " " s
  set s [string trim $s]
  if {[string length $s] > $::vdm::max_len} {
    set s "[string range $s 0 [expr {$::vdm::max_len-4}]]..."
  }
  return $s
}
proc ::vdm::ensure_ends_vdm {s} {
  set t [string trim $s]
  if {![regexp -nocase {VDM\.?\s*$} $t]} { append t " VDM" }
  return $t
}
proc ::vdm::rid_from_link {link} {
  set rid ""
  if {[regexp {/(?:article|articles)/([0-9]{3,})} $link -> rid]} { return $rid }
  if {[regexp {_([0-9]{3,})\.html} $link -> rid]} { return $rid }
  if {[regexp {([0-9]{3,})} $link -> rid]} { return $rid }
  return "??"
}
proc ::vdm::format_line {rid text} {
  if {$rid eq ""} { set rid "??" }
  return "\003$::vdm::id_fg,$::vdm::id_bg\002\[$rid]\002\003$::vdm::quote_fg,$::vdm::quote_bg $text\017"
}
proc ::vdm::send_one {chan rid text} {
  puthelp "PRIVMSG $chan :[::vdm::format_line $rid $text]"
}

# ---------------- HTTP fetch (redirects) ----------------
proc ::vdm::meta_get {meta key} {
  set k [string tolower $key]
  for {set i 0} {$i < [llength $meta]} {incr i 2} {
    if {[string tolower [lindex $meta $i]] eq $k} { return [lindex $meta [expr {$i+1}]] }
  }
  return ""
}

proc ::vdm::fetch {url {depth 0}} {
  if {$depth > $::vdm::max_redirects} { return [list 0 "" {} $url "redirects"] }

  set now [clock milliseconds]
  set delta [expr {$now - $::vdm::last_http_ts}]
  if {$delta < $::vdm::http_cooldown_ms} { after [expr {$::vdm::http_cooldown_ms - $delta}] }

  set eff [::vdm::_rewrite_url $url]
  ::vdm::log_http "GET $eff (depth=$depth)"

  set hdrs [list \
    Accept "*/*" \
    "Accept-Language" "fr-FR,fr;q=0.9,en;q=0.5" \
    "Accept-Encoding" "identity" \
    Connection "close" \
    DNT "1" \
  ]

  if {[catch {::http::geturl $eff -timeout $::vdm::timeout -headers $hdrs -binary 1} token]} {
    return [list 0 "" {} $url "geturl"]
  }

  set code [::http::ncode $token]
  set body [::http::data $token]
  set meta [::http::meta $token]
  ::http::cleanup $token
  set ::vdm::last_http_ts [clock milliseconds]

  set ctype [::vdm::meta_get $meta content-type]
  set loc   [::vdm::meta_get $meta location]
  ::vdm::log_http "HTTP $code bytes=[string length $body] ctype='$ctype' loc='[string range $loc 0 120]'"

  if {$code >= 300 && $code < 400 && $loc ne ""} {
    if {![regexp {^https?://|^httpsvdm://} $loc]} {
      if {[regexp {^(https?://[^/]+)} $url -> origin]} {
        if {[string index $loc 0] ne "/"} { set loc "/$loc" }
        set loc "$origin$loc"
      }
    }
    return [::vdm::fetch $loc [expr {$depth+1}]]
  }

  return [list $code $body $meta $url ""]
}

proc ::vdm::to_text {bytes} {
  if {[catch { set t [encoding convertfrom utf-8 $bytes] }]} { set t $bytes }
  return $t
}

# ---------------- RSS parsing (SANS REGEXP) ----------------
proc ::vdm::rss_items {xml} {
  set x [string map {"\r\n" "\n" "\r" "\n"} $xml]
  set xl [string tolower $x]
  set items {}
  set pos 0
  while {1} {
    set s [string first "<item" $xl $pos]
    if {$s < 0} { break }
    set gt [string first ">" $xl $s]
    if {$gt < 0} { break }
    set e [string first "</item>" $xl $gt]
    if {$e < 0} { break }
    lappend items [string range $x [expr {$gt+1}] [expr {$e-1}]]
    set pos [expr {$e + 7}]
    if {[llength $items] >= 80} { break }
  }
  return $items
}

proc ::vdm::rss_tag {block tag} {
  # Parse <tag ...> ... </tag> (case-insensitive), sans regexp.
  set b $block
  set bl [string tolower $b]
  set tl [string tolower $tag]

  set s [string first "<$tl" $bl 0]
  if {$s < 0} { return "" }

  set gt [string first ">" $bl $s]
  if {$gt < 0} { return "" }

  set close [string first "</$tl>" $bl [expr {$gt+1}]]
  if {$close < 0} { return "" }

  set inner [string range $b [expr {$gt+1}] [expr {$close-1}]]
  set inner [string trim $inner]

  # CDATA (FIX ICI: pas de guillemets, pas de piège d'expression)
  set il [string tolower $inner]
  if {[string first {<![cdata[} $il] == 0} {
    set p [string first {]]>} $inner]
    if {$p > 0} {
      set inner [string range $inner 9 [expr {$p-1}]]
      set inner [string trim $inner]
    } else {
      set inner [string trim [string range $inner 9 end]]
    }
  }

  return $inner
}

proc ::vdm::pick_random_story {xml} {
  set items [::vdm::rss_items $xml]
  ::vdm::log_parse "RSS: items=[llength $items]"

  if {[llength $items] == 0} {
    set snip [string range $xml 0 400]
    regsub -all {\s+} $snip " " snip
    ::vdm::log_sniff "no <item>. snip='$snip'"
    return [list 0 "??" "" "no items"]
  }

  set cand {}
  foreach it $items {
    set link [::vdm::rss_tag $it "link"]
    set guid [::vdm::rss_tag $it "guid"]
    set rid  [::vdm::rid_from_link [expr {$link ne "" ? $link : $guid}]]

    set cenc [::vdm::rss_tag $it "content:encoded"]
    set desc [::vdm::rss_tag $it "description"]
    set tit  [::vdm::rss_tag $it "title"]

    set raw ""
    foreach v [list $cenc $desc $tit] { if {$v ne ""} { set raw $v; break } }
    if {$raw eq ""} { continue }

    set line [::vdm::strip_tags_one_line $raw]
    set line [::vdm::truncate_one_line $line]
    set line [::vdm::ensure_ends_vdm $line]
    if {$line eq ""} { continue }

    lappend cand [list $rid $line]
  }

  if {[llength $cand] == 0} {
    set it0 [lindex $items 0]
    ::vdm::log_sniff "NO USABLE. link.len=[string length [::vdm::rss_tag $it0 link]] title.len=[string length [::vdm::rss_tag $it0 title]] desc.len=[string length [::vdm::rss_tag $it0 description]] cenc.len=[string length [::vdm::rss_tag $it0 content:encoded]]"
    return [list 0 "??" "" "no usable text"]
  }

  # évite répétition immédiate
  set now [clock seconds]
  if {$::vdm::last_rid ne "" && ($now - $::vdm::last_rid_ts) < $::vdm::avoid_same_for_s} {
    set filtered {}
    foreach c $cand { if {[lindex $c 0] ne $::vdm::last_rid} { lappend filtered $c } }
    if {[llength $filtered] > 0} { set cand $filtered }
  }

  set pick [lindex $cand [expr {int(rand()*[llength $cand])}]]
  set rid  [lindex $pick 0]
  set line [lindex $pick 1]

  set ::vdm::last_rid $rid
  set ::vdm::last_rid_ts $now

  return [list 1 $rid $line ""]
}

proc ::vdm::get_random_vdm {} {
  set url "${::vdm::rss_url}?r=[expr {int(rand()*1000000000)}]"
  lassign [::vdm::fetch $url] code body meta final why
  if {$code != 200 || $body eq ""} {
    return [list 0 "??" "" "HTTP $code ($why)"]
  }
  set txt [::vdm::to_text $body]
  return [::vdm::pick_random_story $txt]
}

# ---------------- commande !vdm ----------------
bind pub - "!vdm" ::vdm::cmd

proc ::vdm::cmd {nick uhost hand chan text} {
  if {![channel get $chan VDM]} {
    puthelp "PRIVMSG $chan :Activez +VDM (.chanset $chan +VDM)."
    return
  }
  if {!$::vdm::have_tls && [string match -nocase "https://*" $::vdm::rss_url]} {
    ::vdm::send_one $chan "??" "VDM indisponible : TLS manquant (installe tcl-tls). VDM"
    return
  }

  set ok 0
  set rid "??"
  set story ""
  set why ""

  if {[catch {
    for {set a 0} {$a < $::vdm::max_attempts} {incr a} {
      set delay 0
      if {$a < [llength $::vdm::backoff_s]} { set delay [lindex $::vdm::backoff_s $a] }
      if {$delay > 0} { after [expr {$delay * 1000}] }

      lassign [::vdm::get_random_vdm] ok rid story why
      if {$ok} { break }
      ::vdm::log_err "tentative $a KO: $why (rid=$rid)"
    }
  } err]} {
    ::vdm::log_err "CRASH: $err"
    if {$::vdm::debug >= 2} { putloglev d * "VDM(SNIFF): errorInfo=$::errorInfo" }
    ::vdm::send_one $chan "??" "VDM indisponible (crash parsing). VDM"
    return
  }

  if {!$ok} {
    ::vdm::send_one $chan $rid "VDM indisponible (parsing/site KO). VDM"
    return
  }

  ::vdm::send_one $chan $rid $story
}

putlog "VDM: script v$::vdm::version chargé. Commande !vdm, chanset +VDM requis. Debug: set ::vdm::debug 2"
bind evnt - prerehash ::vdm::unload
########################################################################################
