##############################################################################
# meteo.tcl — v1.3.2 (BMP-ONLY ICONS + RETRIES + LESS TIMEOUT PAIN)
#
# Commande : !meteo <ville>
# Chanset  : +meteo
#
# Objectifs :
#  - icône AVANT la température (comme mediabot) ✅
#  - PAS d’émojis 4-bytes (U+1Fxxx) => uniquement BMP (U+0000..U+FFFF) ✅
#    (ça évite les "ð", "�", affichage pourri sur certains Eggdrops/Tcl)
#  - retries sans flood (1 seul message d’erreur final) ✅
#  - best-effort "Emplacement : ..." sans coords (optionnel) ✅
#
# CHANGELOG
# 1.3.2
#  - IMPORTANT: sortie 100% BMP (☀ ☁ ☂ ❄ ⚡ ░ ● ☵ ➤ ☔) -> fini les "ð"/régressions emoji
#  - retries + rotation d’hôtes (wttr.in / v2.wttr.in) pour réduire les timeouts
#  - timeouts plus serrés + backoff
#
##############################################################################

if { [info commands ::meteo::unload] eq "::meteo::unload" } { ::meteo::unload }
if { [lindex [split $::version] 1] < 1080404 } { putloglev o * "\00304\[meteo - erreur\]\003 Eggdrop trop ancien: \00304${::version}\003 (>= 1.8.4 requis). Chargement annulé."; return }
if { [catch { package require Tcl 8.6 }] } { putloglev o * "\00304\[meteo - erreur\]\003 Tcl 8.6+ requis. Version actuelle: \00304${::tcl_version}\003. Chargement annulé."; return }
if { [catch { package require msgcat 1.6.0 }] } { putloglev o * "\00304\[meteo - erreur\]\003 msgcat 1.6.0+ requis. Chargement annulé."; return }
if { [catch { package require tls } terr] } { putloglev o * "\00304\[meteo - erreur\]\003 package 'tls' requis. Erreur: \00304$terr\003. Chargement annulé."; return }

package provide meteo 1.3.2

namespace eval ::meteo {
  catch { encoding system utf-8 }
  catch { setudef flag meteo }

  # Debug Eggdrop: .console +d
  variable debug 1

  # Rotation d’hôtes (wttr.in a parfois des trous)
  variable hosts {wttr.in v2.wttr.in}

  variable https_port 443
  variable http_port  80

  # Format SANS emoji : on injecte nos icônes BMP nous-mêmes ensuite
  # %l location, %t temp, %f feelslike, %h humidity, %w wind, %p precip
  variable format "%l: %t (ressenti %f) | %h | %w | %p"

  # --- Icônes BMP-only (PAS de U+1Fxxx) ---
  # métriques
  variable ICON_HUM  "\u2635"  ;# ☵
  variable ICON_WIND "\u27A4"  ;# ➤
  variable ICON_RAIN "\u2614"  ;# ☔

  # météo (condition -> icône) : BMP-only
  variable ICON_SUN   "\u2600"  ;# ☀
  variable ICON_CLOUD "\u2601"  ;# ☁
  variable ICON_RAIN2 "\u2602"  ;# ☂
  variable ICON_SNOW  "\u2744"  ;# ❄
  variable ICON_STORM "\u26A1"  ;# ⚡
  variable ICON_FOG   "\u2591"  ;# ░
  variable ICON_UNK   "\u25CF"  ;# ●

  # Menz: Emplacement exact (best effort) + nettoyage
  variable show_exact_location 1
  variable location_max_admin_parts 2

  # Réseau / retries (sans flood)
  variable max_attempts 4
  variable backoff {0 2 4 8}       ;# secondes entre tentatives (par requête)
  variable connect_timeout 5        ;# secondes
  variable io_timeout      10       ;# secondes

  variable cache_ttl 300
  variable max_len   350
  variable user_agent "eggdrop-meteo/1.3.2 (+https://teuk.org)"

  # cache(key)=dict(ts text)
  variable cache
  array set cache {}

  # req(id)=dict{chan key loc fh state buf timer scheme redirects host_idx host_override path_override phase pending_text attempts}
  variable req
  array set req {}
  variable seq 0
}

proc ::meteo::dbg {msg} { if {$::meteo::debug} { putloglev d * "\[meteo\] $msg" } }

# Envoi simple (on reste en BMP-only => pas de galère d’UTF-8 4 bytes)
proc ::meteo::sendmsg {chan msg} {
  putserv "PRIVMSG $chan :$msg"
}

proc ::meteo::sanitize {txt} {
  regsub -all {\r|\n} $txt " " txt
  set txt [string trim $txt]
  regsub -all {♦\s*} $txt "" txt
  regsub -all {\s+} $txt " " txt
  while {[regsub { \| \| } $txt " | " txt]} {}
  regsub -all {\s*\|\s*} $txt " | " txt
  set txt [string trim $txt]
  if {[string length $txt] > $::meteo::max_len} {
    set txt "[string range $txt 0 [expr {$::meteo::max_len - 4}]]..."
  }
  return $txt
}

proc ::meteo::cache_get {key} {
  if {![info exists ::meteo::cache($key)]} { return "" }
  set d $::meteo::cache($key)
  set ts [dict get $d ts]
  if {[clock seconds] - $ts > $::meteo::cache_ttl} { unset ::meteo::cache($key); return "" }
  return [dict get $d text]
}
proc ::meteo::cache_set {key text} { set ::meteo::cache($key) [dict create ts [clock seconds] text $text] }

proc ::meteo::urlencode_utf8 {s} {
  # OK même si le texte contient des trucs non-BMP -> on n’en envoie pas côté sortie,
  # mais côté URL c’est safe.
  set b   [encoding convertto utf-8 $s]
  set hex [binary encode hex $b]
  set out ""
  for {set i 0} {$i < [string length $hex]} {incr i 2} {
    set bytehex [string range $hex $i [expr {$i+1}]]
    scan $bytehex %x u
    if {($u>=48 && $u<=57) || ($u>=65 && $u<=90) || ($u>=97 && $u<=122) || $u==45 || $u==46 || $u==95 || $u==126} {
      append out [format %c $u]
    } else {
      append out %[string toupper $bytehex]
    }
  }
  return $out
}

# --- Menz: location ---
proc ::meteo::strip_coords {s} { regsub {\s*\[[0-9\.\-]+\s*,\s*[0-9\.\-]+\]\s*$} $s "" s; return [string trim $s] }
proc ::meteo::prettify_location {full} {
  set full [::meteo::strip_coords $full]
  set full [string trim $full]
  if {$full eq ""} { return "" }

  set parts {}
  foreach p [split $full ","] {
    set p [string trim $p]
    if {$p eq ""} { continue }
    if {[string match -nocase "France métropolitaine" $p]} { continue }
    if {[regexp {^\d{4,6}$} $p]} { continue }
    set seen 0
    foreach q $parts { if {[string equal -nocase $q $p]} { set seen 1; break } }
    if {!$seen} { lappend parts $p }
  }
  if {[llength $parts] == 0} { return "" }
  if {[llength $parts] == 1} { return [lindex $parts 0] }

  set city [lindex $parts 0]
  set country [lindex $parts end]
  set mid [lrange $parts 1 end-1]

  set N $::meteo::location_max_admin_parts
  if {$N < 0} { set N 0 }
  if {[llength $mid] > $N} {
    if {$N == 0} { set mid {} } else { set mid [lrange $mid 0 [expr {$N-1}]] }
  }

  set out $city
  foreach m $mid { append out ", $m" }
  if {$country ne ""} { append out ", $country" }
  return $out
}

proc ::meteo::extract_exact_location {html} {
  set html [string map {"\r" ""} $html]
  set full ""
  if {$full eq ""} {
    if {[regexp -nocase {(?:Emplacement|Location)\s*:\s*([^<\n]+)} $html -> full]} { set full [string trim $full] }
  }
  if {$full eq ""} {
    if {[regexp -nocase {<meta\s+name="description"\s+content="[^"]*?\b(?:in|à)\s+([^"]+?)"} $html -> full]} { set full [string trim $full] }
  }
  if {$full ne ""} {
    regsub -all {\s+} $full " " full
    set full [string trim $full]
    set full [::meteo::prettify_location $full]
  }
  return $full
}

# --- Condition -> icon (BMP only) ---
proc ::meteo::cond_icon {cond} {
  set c [string tolower [string trim $cond]]
  if {$c eq ""} { return $::meteo::ICON_UNK }

  if {[regexp {thunder|orage|storm} $c]} { return $::meteo::ICON_STORM }
  if {[regexp {snow|neige|sleet|blizzard|ice|gr\xeale} $c]} { return $::meteo::ICON_SNOW }
  if {[regexp {rain|pluie|drizzle|shower|averses} $c]} { return $::meteo::ICON_RAIN2 }
  if {[regexp {fog|brume|mist|haze|brouillard} $c]} { return $::meteo::ICON_FOG }
  if {[regexp {overcast|cloud|nuage} $c]} { return $::meteo::ICON_CLOUD }
  if {[regexp {sun|clear|ensole|clair} $c]} { return $::meteo::ICON_SUN }

  return $::meteo::ICON_UNK
}

# --- retry helpers (sans flood) ---
proc ::meteo::final_fail {id reason} {
  if {![info exists ::meteo::req($id)]} { return }
  set d $::meteo::req($id)
  set chan [dict get $d chan]
  catch { unset ::meteo::req($id) }
  ::meteo::dbg "FINAL FAIL id=$id reason=$reason"
  ::meteo::sendmsg $chan "Service météo temporairement indisponible (timeout), réessaie dans quelques secondes."
}

proc ::meteo::schedule_retry {id reason} {
  if {![info exists ::meteo::req($id)]} { return }
  set d $::meteo::req($id)

  set attempt [dict get $d attempts]
  incr attempt
  dict set d attempts $attempt
  set ::meteo::req($id) $d

  ::meteo::dbg "retry attempt=$attempt reason=$reason id=$id phase=[dict get $d phase]"

  if {$attempt >= $::meteo::max_attempts} {
    ::meteo::final_fail $id $reason
    return
  }

  set delay 2
  if {$attempt < [llength $::meteo::backoff]} { set delay [lindex $::meteo::backoff $attempt] }
  utimer $delay [list ::meteo::restart_phase $id]
}

proc ::meteo::restart_phase {id} {
  if {![info exists ::meteo::req($id)]} { return }
  set d $::meteo::req($id)

  # clean old fh/timer if any
  if {[dict exists $d timer]} { catch { killutimer [dict get $d timer] } }
  if {[dict exists $d fh] && [dict get $d fh] ne ""} {
    set fh [dict get $d fh]
    catch { fileevent $fh readable "" }
    catch { fileevent $fh writable "" }
    catch { close $fh }
  }

  # rotation d'hôte à chaque retry
  set idx [dict get $d host_idx]
  set idx [expr {($idx + 1) % [llength $::meteo::hosts]}]
  dict set d host_idx $idx

  dict set d fh ""
  dict set d buf ""
  dict set d redirects 0
  dict set d host_override ""
  dict set d path_override ""
  set ::meteo::req($id) $d

  ::meteo::restart_request $id "https"
}

# --- timeouts ---
proc ::meteo::timeout {id where} {
  if {![info exists ::meteo::req($id)]} { return }
  set d $::meteo::req($id)
  set fh [dict get $d fh]
  catch { fileevent $fh readable "" }
  catch { fileevent $fh writable "" }
  catch { close $fh }
  dict set d fh ""
  set ::meteo::req($id) $d
  ::meteo::schedule_retry $id "timeout/$where"
}

proc ::meteo::set_timeout {id seconds where} {
  if {![info exists ::meteo::req($id)]} { return }
  set d $::meteo::req($id)
  if {[dict exists $d timer]} { catch { killutimer [dict get $d timer] } }
  set t [utimer $seconds [list ::meteo::timeout $id $where]]
  dict set d timer $t
  set ::meteo::req($id) $d
}

# --- network core ---
proc ::meteo::pick_host {d} {
  if {[dict exists $d host_override] && [dict get $d host_override] ne ""} {
    return [dict get $d host_override]
  }
  set idx [dict get $d host_idx]
  return [lindex $::meteo::hosts $idx]
}

proc ::meteo::restart_request {id scheme} {
  if {![info exists ::meteo::req($id)]} { return }
  set d $::meteo::req($id)

  set host [::meteo::pick_host $d]
  set port [expr {$scheme eq "https" ? $::meteo::https_port : $::meteo::http_port}]
  set phase [dict get $d phase]
  ::meteo::dbg "restart_request id=$id scheme=$scheme host=$host port=$port phase=$phase"

  set fh ""
  if {[catch { set fh [socket -async $host $port] } e]} {
    ::meteo::dbg "socket failed id=$id err=$e"
    ::meteo::schedule_retry $id "socket"
    return
  }

  fconfigure $fh -blocking 0 -translation binary -encoding binary -buffering none
  dict set d fh $fh
  dict set d scheme $scheme
  dict set d state "connecting"
  dict set d buf ""
  set ::meteo::req($id) $d

  ::meteo::set_timeout $id $::meteo::connect_timeout "connect"
  fileevent $fh writable [list ::meteo::on_writable $id]
}

proc ::meteo::build_path {d} {
  if {[dict exists $d path_override] && [dict get $d path_override] ne ""} { return [dict get $d path_override] }
  set loc [dict get $d loc]
  set enc_loc [::meteo::urlencode_utf8 $loc]
  set phase [dict get $d phase]
  if {$phase eq "html"} { return "/$enc_loc" }
  if {$phase eq "cond"} { return "/$enc_loc?format=%25C&m" }  ;# condition text only
  set enc_fmt [::meteo::urlencode_utf8 $::meteo::format]
  return "/$enc_loc?format=$enc_fmt&m"
}

proc ::meteo::on_writable {id} {
  if {![info exists ::meteo::req($id)]} { return }
  set d $::meteo::req($id)
  set fh [dict get $d fh]
  set scheme [dict get $d scheme]
  set phase  [dict get $d phase]

  catch { fileevent $fh writable "" }

  set host [::meteo::pick_host $d]

  if {$scheme eq "https"} {
    if {[catch { ::tls::import $fh -server 0 -servername $host -tls1 1 } e]} {
      ::meteo::dbg "TLS import failed id=$id err=$e"
      catch { close $fh }
      ::meteo::schedule_retry $id "tls"
      return
    }
  }

  set path [::meteo::build_path $d]
  set accept [expr {$phase eq "html" ? "text/html" : "text/plain"}]
  set req "GET $path HTTP/1.1\r\nHost: $host\r\nUser-Agent: $::meteo::user_agent\r\nAccept: $accept\r\nAccept-Language: fr-FR,fr;q=0.9,en;q=0.5\r\nConnection: close\r\n\r\n"

  ::meteo::dbg "sending id=$id phase=$phase path=$path"
  if {[catch { puts -nonewline $fh $req; flush $fh } e2]} {
    ::meteo::dbg "write failed id=$id err=$e2"
    catch { close $fh }
    ::meteo::schedule_retry $id "write"
    return
  }

  dict set d state "reading"
  set ::meteo::req($id) $d
  ::meteo::set_timeout $id $::meteo::io_timeout "read"
  fileevent $fh readable [list ::meteo::on_readable $id]
}

proc ::meteo::on_readable {id} {
  if {![info exists ::meteo::req($id)]} { return }
  set d $::meteo::req($id)
  set fh [dict get $d fh]

  set chunk ""
  if {[catch { set chunk [read $fh] } er]} {
    ::meteo::dbg "read error id=$id err=$er"
    catch { close $fh }
    ::meteo::schedule_retry $id "read"
    return
  }

  if {$chunk ne ""} {
    dict set d buf "[dict get $d buf]$chunk"
    set ::meteo::req($id) $d
  }

  if {[eof $fh]} {
    catch { fileevent $fh readable "" }
    catch { close $fh }
    ::meteo::finish $id
  }
}

proc ::meteo::parse_location {headers} {
  foreach line [split $headers "\n"] {
    set line [string trim $line]
    if {[regexp -nocase {^location:\s*(.+)$} $line -> loc]} { return [string trim $loc] }
  }
  return ""
}

proc ::meteo::apply_redirect {id location} {
  set d $::meteo::req($id)
  set redirects [dict get $d redirects]
  if {$redirects >= 3} { return 0 }
  dict set d redirects [expr {$redirects + 1}]
  dict set d host_override ""
  dict set d path_override ""

  if {[regexp -nocase {^(https?)://([^/]+)(/.*)$} $location -> scheme host path]} {
    dict set d host_override $host
    dict set d path_override $path
    set ::meteo::req($id) $d
    ::meteo::restart_request $id $scheme
    return 1
  } elseif {[string match "/*" $location]} {
    dict set d path_override $location
    set ::meteo::req($id) $d
    ::meteo::restart_request $id [dict get $d scheme]
    return 1
  }

  set ::meteo::req($id) $d
  return 0
}

proc ::meteo::finish {id} {
  if {![info exists ::meteo::req($id)]} { return }
  set d $::meteo::req($id)
  if {[dict exists $d timer]} { catch { killutimer [dict get $d timer] } }

  set raw [dict get $d buf]
  set phase [dict get $d phase]

  set idx [string first "\r\n\r\n" $raw]
  if {$idx < 0} { set idx [string first "\n\n" $raw] }
  if {$idx < 0} { ::meteo::schedule_retry $id "badresp"; return }

  set headers [string range $raw 0 [expr {$idx-1}]]
  set body    [string range $raw [expr {$idx+4}] end]

  set code 0
  foreach line [split $headers "\n"] {
    set line [string trim $line]
    if {[regexp {^HTTP/[^ ]+\s+(\d+)} $line -> c]} { set code $c }
  }

  if {$code == 301 || $code == 302 || $code == 307 || $code == 308} {
    set loc [::meteo::parse_location $headers]
    if {$loc ne "" && [::meteo::apply_redirect $id $loc]} { return }
    ::meteo::schedule_retry $id "redirect"
    return
  }
  if {$code < 200 || $code >= 300} { ::meteo::schedule_retry $id "http$code"; return }

  set txt ""
  if {[catch { set txt [encoding convertfrom utf-8 $body] }]} { set txt $body }
  set txt [::meteo::sanitize $txt]
  if {$txt eq ""} { ::meteo::schedule_retry $id "empty"; return }

  set chan [dict get $d chan]
  set key  [dict get $d key]

  # --- Phase TXT: on injecte les icônes métriques et on chaîne COND ---
  if {$phase eq "txt"} {
    # inject HUM/WIND/RAIN (BMP-only)
    if {[regexp {^(.*)\|\s*([^|]+)\|\s*([^|]+)\|\s*([^|]+)$} $txt -> head hum wind rain]} {
      set head [string trim $head]
      set hum  [string trim $hum]
      set wind [string trim $wind]
      set rain [string trim $rain]
      set txt "$head | $::meteo::ICON_HUM $hum | $::meteo::ICON_WIND $wind | $::meteo::ICON_RAIN $rain"
    }

    ::meteo::cache_set $key $txt

    dict set d phase "cond"
    dict set d pending_text $txt
    set ::meteo::req($id) $d
    ::meteo::restart_request $id "https"
    return
  }

  # --- Phase COND: on met une icône AVANT la temp ---
  if {$phase eq "cond"} {
    set icon [::meteo::cond_icon $txt]
    set pending [dict get $d pending_text]

    # insère juste après "<loc>: "
    if {[regexp {^([^:]+:\s*)(.+)$} $pending -> pfx rest]} {
      set pending "${pfx}${icon} ${rest}"
    } else {
      set pending "${icon} $pending"
    }

    if {$::meteo::show_exact_location} {
      dict set d phase "html"
      dict set d pending_text $pending
      set ::meteo::req($id) $d
      ::meteo::restart_request $id "https"
      return
    }

    unset ::meteo::req($id)
    ::meteo::sendmsg $chan $pending
    return
  }

  # --- Phase HTML: emplacement exact (optionnel) ---
  set exact [::meteo::extract_exact_location $txt]
  set pending [dict get $d pending_text]
  unset ::meteo::req($id)
  if {$exact ne ""} { ::meteo::sendmsg $chan "Emplacement : $exact" }
  ::meteo::sendmsg $chan $pending
}

proc ::meteo::fetch {chan location} {
  set loc [string trim $location]
  if {$loc eq ""} { ::meteo::sendmsg $chan "Utilisation : !meteo <ville> (ex: !meteo Tunis / !meteo \"paris usa\")"; return }

  set key [string tolower $loc]
  set cached [::meteo::cache_get $key]
  if {$cached ne ""} { ::meteo::sendmsg $chan $cached; return }

  # anti-parallélisme par key
  foreach id [array names ::meteo::req] {
    if {[dict get $::meteo::req($id) key] eq $key} {
      ::meteo::sendmsg $chan "Requête météo déjà en cours pour « $loc ». Réessaie dans un instant."
      return
    }
  }

  set id [incr ::meteo::seq]
  set ::meteo::req($id) [dict create \
    chan $chan key $key loc $loc fh "" state "init" buf "" timer "" scheme "https" redirects 0 \
    host_idx 0 host_override "" path_override "" phase "txt" pending_text "" attempts 0 \
  ]
  ::meteo::restart_request $id "https"
}

proc ::meteo::pubm {nick uhost hand chan text} {
  set enabled 0
  catch { set enabled [channel get $chan meteo] }
  if {!$enabled} { return 0 }

  set location ""
  if {![regexp -nocase {^!meteo\s+(.+)$} [string trim $text] -> location]} { return 0 }
  ::meteo::fetch $chan [string trim $location]
  return 1
}

bind pubm - "*" ::meteo::pubm

proc ::meteo::unload {} {
  catch { unbind pubm - "*" ::meteo::pubm }
  foreach id [array names ::meteo::req] {
    set d $::meteo::req($id)
    if {[dict exists $d timer]} { catch { killutimer [dict get $d timer] } }
    if {[dict exists $d fh] && [dict get $d fh] ne ""} {
      set fh [dict get $d fh]
      catch { fileevent $fh readable "" }
      catch { fileevent $fh writable "" }
      catch { close $fh }
    }
    catch { unset ::meteo::req($id) }
  }
  catch { unset ::meteo::cache }
  putloglev o * "\[meteo\] unloaded"
}

putloglev o * "\[meteo\] loaded v1.3.2 — .chanset #chan +meteo — !meteo <ville> (icône avant temp + retries sans flood, BMP-only)"
::meteo::dbg "debug ON (.console +d) / off: set ::meteo::debug 0"
