###############################################################################
# ddg.tcl
# v1.0.7  ©2025 Te[u]K
# Recherche DuckDuckGo pour Eggdrop (!ddg) + raccourcisseur short.io (optionnel)
###############################################################################

if {[info commands ::teuk::ddg::unload] eq "::teuk::ddg::unload"} { ::teuk::ddg::unload }
if { [lindex [split $::version] 1] < 1080404 } { putloglev o * "\00304[ddg - erreur]\003 Eggdrop trop ancien: \00304${::version}\003 (>= 1.8.4 requis). Chargement annulé."; return }
if { [catch { package require Tcl 8.6 } e] } { putloglev o * "\00304[ddg - erreur]\003 Tcl 8.6+ requis. Version actuelle: \00304${::tcl_version}\003. Chargement annulé."; return }
if { [catch { package require http } e] } { putloglev o * "\00304[ddg - erreur]\003 package http requis : \00304$e\003"; return }
if { [catch { package require tls } e] } { putloglev o * "\00304[ddg - erreur]\003 package tls requis pour HTTPS : \00304$e\003"; return }

# Force UTF-8 (interne Tcl + stdout/stderr)
catch { encoding system utf-8 }
catch { fconfigure stdout -encoding utf-8 }
catch { fconfigure stderr -encoding utf-8 }

package provide ddg 1.0.7

###############################################################################
# NAMESPACE
###############################################################################

namespace eval ::teuk::ddg {
    variable version "1.0.7"
    variable debug   3          ;# 0=silence, 1=info, 2=debug, 3=trace

    variable base_url     "https://duckduckgo.com/html/"
    variable max_results  4
    variable max_inline   8
    variable max_snippet  160

    # inline display
    variable inline_sep   " \00307|\003 "

    variable color_link  "\00312"
    variable color_reset "\003"

    # Bullet: ne dépend pas de l'encodage du fichier source
    variable bullet "\u2022"

    # short.io
    variable short_api_key ""
    variable short_domain  ""
    variable use_shortio   0

    # check HEAD désactivé par défaut (trop lent)
    variable short_check   0
    variable short_timeout 2500

    variable sleep_ms 0

    catch { setudef flag ddg }
}

###############################################################################
# LOGGING
###############################################################################

proc ::teuk::ddg::log {lvl tag msg} {
    if {$::teuk::ddg::debug >= $lvl} { putlog "ddg($tag): $msg" }
}
proc ::teuk::ddg::log1 {msg} { ::teuk::ddg::log 1 INFO  $msg }
proc ::teuk::ddg::log2 {msg} { ::teuk::ddg::log 2 DEBUG $msg }
proc ::teuk::ddg::log3 {msg} { ::teuk::ddg::log 3 TRACE $msg }

proc ::teuk::ddg::_trace_loop {label iters ms} {
    ::teuk::ddg::log3 "loop:$label iters=$iters ms=$ms"
}

###############################################################################
# HTTP INIT
###############################################################################

::teuk::ddg::log1 "http pkg: [package provide http]"
::teuk::ddg::log1 "tls pkg:  [package provide tls]"

catch { ::http::config -useragent "Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:146.0) Gecko/20100101 Firefox/146.0" }
catch { ::tls::init -tls1 1 }

###############################################################################
# UNLOAD
###############################################################################

proc ::teuk::ddg::unload {args} {
    set ns [namespace current]
    ::teuk::ddg::log1 "nettoyage namespace $ns"

    foreach b [binds *] {
        if {[llength $b] < 5} { continue }
        set cmd [lindex $b 4]
        if {[string match "${ns}::*" $cmd]} {
            catch { unbind [lindex $b 0] [lindex $b 1] [lindex $b 2] $cmd }
        }
    }

    catch { ::http::unregister httpsddg }
    catch { namespace delete $ns }
}

###############################################################################
# TLS SOCKET  (template Menz inchangé)
###############################################################################

proc ::teuk::ddg::_tls_socket {args} {
    set async 0; set host ""; set port ""

    set i 0; set n [llength $args]
    while {$i < $n} {
        set a [lindex $args $i]
        if {![string match -* $a]} {
            if {$host eq ""} {
                set host $a
            } elseif {$port eq ""} {
                set port $a
            }
        }
        incr i
    }

    if {$host eq "" || $port eq ""} { error "_tls_socket: args invalides" }

    if {[catch { set s [::socket $host $port] } e]} {
        return -code error "socket error $host:$port -> $e"
    }

    catch { fconfigure $s -translation binary -encoding binary -buffering none }
    if {$async} { catch { fconfigure $s -blocking 0 } }

    if {[catch { ::tls::import $s -servername $host } e]} {
        catch { close $s }
        return -code error "tls::import a échoué: $e"
    }

    return $s
}

catch { ::http::register httpsddg 443 ::teuk::ddg::_tls_socket }
::teuk::ddg::log2 "http::register httpsddg OK"

###############################################################################
# URL
###############################################################################

proc ::teuk::ddg::_rewrite_url {url} {
    if {[string match -nocase "https://*" $url]} {
        return [string map {"https://" "httpsddg://"} $url]
    }
    return $url
}

###############################################################################
# ENTITY / HTML / URL DECODERS
###############################################################################

proc ::teuk::ddg::_decode_entities {s} {
    set res $s
    set res [string map {
        "&nbsp;" " "
        "&amp;"  "&"
        "&lt;"   "<"
        "&gt;"   ">"
        "&quot;" "\""
        "&#39;"  "'"
    } $res]

    while {[regexp -nocase {&#x([0-9a-f]+);?} $res -> hex]} {
        set ch [format %c 0x$hex]
        regsub -nocase "&#x$hex;?" $res $ch res
    }

    while {[regexp {&#([0-9]+);?} $res -> dec]} {
        if {$dec > 1114111} break
        set ch [format %c $dec]
        regsub "&#$dec;?" $res $ch res
    }

    return $res
}

proc ::teuk::ddg::_strip_html {s} {
    regsub -all {<[^>]+>} $s "" s
    set s [::teuk::ddg::_decode_entities $s]
    # IMPORTANT: avoid \n / \r breaking PRIVMSG lines
    regsub -all {\s+} $s " " s
    return [string trim $s]
}

# FIX RACINE UTF-8:
# - on reconstruit des OCTETS (bytearray)
# - puis on décode UTF-8 une fois
proc ::teuk::ddg::_urldecode {s} {
    set t0 [clock milliseconds]

    # '+' -> espace
    regsub -all {\+} $s " " s

    set bytes ""
    set i 0
    set len [string length $s]

    while {$i < $len} {
        set c [string index $s $i]
        if {$c eq "%" && $i+2 < $len} {
            set h [string range $s [expr {$i+1}] [expr {$i+2}]]
            if {[string is xdigit -strict $h]} {
                append bytes [binary format H2 $h]
                incr i 3
                continue
            }
        }
        append bytes $c
        incr i
    }

    if {[catch { set out [encoding convertfrom utf-8 $bytes] }]} {
        set out $bytes
    }

    if {$::teuk::ddg::debug >= 3} {
        ::teuk::ddg::_trace_loop "urldecode" $len [expr {[clock milliseconds]-$t0}]
    }
    return $out
}

# Return 1 if it's a DDG ad/tracker redirect (the huge y.js? stuff)
proc ::teuk::ddg::_is_ddg_ad_url {url} {
    return [regexp -nocase {^https?://duckduckgo\.com/y\.js(\?|$)} $url]
}

# Keep URL clickable but short: remove query string + fragment
proc ::teuk::ddg::_compact_url {url} {
    set u [string trim $url]
    regsub {#.*$} $u "" u
    regsub {\?.*$} $u "" u
    return $u
}

# Shorten a title for inline display
proc ::teuk::ddg::_short_title {t {max 48}} {
    set s [string trim $t]
    regsub -all {\s+} $s " " s
    if {[string length $s] > $max} {
        return "[string range $s 0 [expr {$max-4}]]..."
    }
    return $s
}

proc ::teuk::ddg::_short_snippet {s {max 90}} {
    set t [string trim $s]
    if {$t eq "" || $t eq "-"} { return "-" }
    regsub -all {\s+} $t " " t
    # Avoid trailing punctuation spam
    # NOTE: do NOT escape the em-dash (—) in Tcl regex; "\—" breaks regexp compilation.
    regsub {[\s\.,;:\-—]+$} $t "" t
    if {[string length $t] > $max} {
        return "[string range $t 0 [expr {$max-4}]]..."
    }
    return $t
}

proc ::teuk::ddg::_normalize_url {url} {
    set url [string trim $url]

    if {[string match {//*} $url]} {
        set url "https:$url"
    }

    # redirect classique DDG
    if {[regexp {duckduckgo\.com/l/\?uddg=([^&]+)} $url -> enc]} {
        set dec [::teuk::ddg::_urldecode $enc]
        if {$dec ne ""} { return $dec }
    }

    # parfois param u3 (ads)
    if {[regexp {[\?&]u3=([^&]+)} $url -> enc3]} {
        set dec3 [::teuk::ddg::_urldecode $enc3]
        if {$dec3 ne ""} { return $dec3 }
    }

    return $url
}

# Ensure URL always fits in the IRC line by shrinking snippet dynamically
proc ::teuk::ddg::_fit_snippet {title snip url {target_len 420}} {
    set snip [string trim $snip]
    if {$snip eq "-" || $snip eq ""} { return "-" }

    set base "$::teuk::ddg::bullet $title — "
    set tail " $::teuk::ddg::color_link\037$url\037$::teuk::ddg::color_reset"

    set maxSnipLen [expr {$target_len - [string length $base] - [string length $tail]}]
    if {$maxSnipLen < 10} { return "-" }

    if {[string length $snip] > $maxSnipLen} {
        set snip "[string range $snip 0 [expr {$maxSnipLen-4}]]..."
    }
    return $snip
}

###############################################################################
# SHORT.IO (original 1.0.6 kept)
###############################################################################

proc ::teuk::ddg::_check_head {url} {
    set u [::teuk::ddg::_rewrite_url $url]
    if {[catch { set t [::http::geturl $u -method HEAD -timeout $::teuk::ddg::short_timeout] } e]} {
        return [list 0 $e]
    }
    set c [::http::ncode $t]
    ::http::cleanup $t
    if {$c >= 200 && $c < 400} { return [list 1 $c] }
    return [list 0 $c]
}

proc ::teuk::ddg::_shorten_url {url} {
    if {!$::teuk::ddg::use_shortio} { return $url }
    if {$::teuk::ddg::short_api_key eq ""} { return $url }
    if {$::teuk::ddg::short_domain eq ""} { return $url }

    set api "https://api.short.io/links"
    set api [::teuk::ddg::_rewrite_url $api]
    set payload "{\"domain\":\"$::teuk::ddg::short_domain\",\"originalURL\":\"$url\",\"allowDuplicates\":false}"

    set headers [list \
        accept "application/json" \
        Content-Type "application/json; charset=utf-8" \
        authorization $::teuk::ddg::short_api_key \
    ]

    ::teuk::ddg::log2 "short.io: POST $api pour $url (domain=$::teuk::ddg::short_domain)"

    if {[catch {
        set tok [::http::geturl $api -method POST -headers $headers -type "application/json" -query $payload -timeout 8000]
    } e]} {
        ::teuk::ddg::log3 "short.io: POST error -> $e"
        return $url
    }

    set code [::http::ncode $tok]
    set body [::http::data $tok]
    ::http::cleanup $tok

    if {$code < 200 || $code >= 300} {
        ::teuk::ddg::log3 "short.io: HTTP $code body=$body"
        return $url
    }

    if {![regexp -nocase {\"shortURL\"\s*:\s*\"([^\"]+)\"} $body -> short]} {
        ::teuk::ddg::log3 "short.io: shortURL introuvable"
        return $url
    }

    if {$::teuk::ddg::short_check} {
        lassign [::teuk::ddg::_check_head $short] ok why
        if {!$ok} {
            ::teuk::ddg::log3 "short.io: HEAD check fail for $short -> $why"
            return $url
        }
    }

    return $short
}

###############################################################################
# FETCH URL
###############################################################################

proc ::teuk::ddg::_fetch_url {url {depth 0}} {
    if {$depth > 5} { error "trop de redirections" }

    set u [::teuk::ddg::_rewrite_url $url]
    ::teuk::ddg::log2 "GET $u (depth=$depth)"

    set tok ""
    if {[catch {
        set tok [::http::geturl $u -timeout 12000]
    } e]} {
        error "http::geturl: $e"
    }

    set code [::http::ncode $tok]
    set meta [::http::meta $tok]
    set body [::http::data $tok]
    ::http::cleanup $tok

    ::teuk::ddg::log1 "HTTP $code pour $u (status=ok len=[string length $body])"

    if {$::teuk::ddg::debug >= 3} {
        set preview [string range $body 0 400]
        ::teuk::ddg::log3 "BODY(0..400): $preview"
    }

    if {$code >= 300 && $code < 400} {
        set loc ""
        foreach {k v} $meta {
            if {[string equal -nocase $k "Location"]} {
                set loc $v
                break
            }
        }
        if {$loc eq ""} { error "redirection sans Location" }
        ::teuk::ddg::log2 "redirection -> $loc"
        return [::teuk::ddg::_fetch_url $loc [expr {$depth+1}]]
    }

    if {$code < 200 || $code >= 300} {
        error "HTTP $code"
    }

    return $body
}

###############################################################################
# PARSE HTML (DuckDuckGo html/)
###############################################################################

proc ::teuk::ddg::parse_results {html {max 4}} {
    set limit $::teuk::ddg::max_snippet
    set lenHtml [string length $html]

    set positions [regexp -all -inline -indices -nocase -- {result__a} $html]
    ::teuk::ddg::log2 "parse_results: tokens 'result__a' trouvés=[llength $positions]"

    set results {}
    set idx 0

    set tloop [clock milliseconds]
    set iters 0
    foreach pair $positions {
        incr iters
        lassign $pair start end
        if {[llength $results] >= $max} { break }

        # window around the match
        set winStart [expr {$start - 800}]
        if {$winStart < 0} { set winStart 0 }
        set winEnd [expr {$start + 2500}]
        if {$winEnd >= $lenHtml} { set winEnd [expr {$lenHtml - 1}] }

        set window [string range $html $winStart $winEnd]

        set href ""
        set titleHtml ""
        if {![regexp -nocase -- {<a[^>]+class="[^"]*result__a[^"]*"[^>]+href="([^"]+)"[^>]*>(.*?)</a>} $window -> href titleHtml]} {
            continue
        }

        set url [::teuk::ddg::_normalize_url $href]
        if {![string match -nocase "http://*" $url] && ![string match -nocase "https://*" $url]} {
            continue
        }

        set title [::teuk::ddg::_strip_html $titleHtml]
        if {$title eq ""} { continue }

        set snip "-"
        if {[regexp -nocase -- {class="[^"]*result__snippet[^"]*"[^>]*>(.*?)</} $window -> snHtml]} {
            set snip [::teuk::ddg::_strip_html $snHtml]
        }

        if {[string length $snip] > $limit} {
            set snip "[string range $snip 0 [expr {$limit-4}]]..."
        }

        incr idx
        lappend results [list $title $snip $url]
        ::teuk::ddg::log3 "parse_results: résultat #$idx -> title='$title' url='$url'"
    }

    if {$::teuk::ddg::debug >= 3} {
        ::teuk::ddg::_trace_loop "parse_positions" $iters [expr {[clock milliseconds]-$tloop}]
    }

    ::teuk::ddg::log1 "parse_results: [llength $results] résultat(s) trouvé(s)"
    return $results
}

###############################################################################
# SEND IRC
###############################################################################

proc ::teuk::ddg::_send_lines {target lines} {
    set t0 [clock milliseconds]
    set it 0
    foreach l $lines {
        incr it
        if {$l eq ""} { continue }
        puthelp "PRIVMSG $target :$l"
        after $::teuk::ddg::sleep_ms
    }
    if {$::teuk::ddg::debug >= 3} {
        ::teuk::ddg::_trace_loop "send_lines" $it [expr {[clock milliseconds]-$t0}]
    }
}

proc ::teuk::ddg::_send_inline {target items} {
    # items are chunks "Title URL"
    set out ""
    foreach it $items {
        if {$it eq ""} { continue }

        if {$out eq ""} {
            set cand $it
        } else {
            set cand "${out}$::teuk::ddg::inline_sep$it"
        }

        if {[string length $cand] > 420} {
            if {$out ne ""} { puthelp "PRIVMSG $target :$out" }
            after $::teuk::ddg::sleep_ms
            set out $it
        } else {
            set out $cand
        }
    }
    if {$out ne ""} { puthelp "PRIVMSG $target :$out" }
}

###############################################################################
# PIPELINE
###############################################################################

proc ::teuk::ddg::do_search {nick target query {mode "lines"}} {
    set q [string trim $query]
    if {$q eq ""} {
        puthelp "NOTICE $nick :Syntaxe: !ddg <recherche>  |  !ddg -full <recherche>  |  !ddg -full -inline <recherche>"
        return
    }

    ::teuk::ddg::log1 "CMD !ddg par $nick sur $target : '$q'"

    set qenc [::http::formatQuery q $q]
    set url "${::teuk::ddg::base_url}?$qenc&kl=fr-fr&kp=-2"
    ::teuk::ddg::log2 "URL: $url"

    if {[catch { set body [::teuk::ddg::_fetch_url $url] } e]} {
        ::teuk::ddg::log1 "fetch error: $e"
        puthelp "PRIVMSG $target :Erreur DDG: $e"
        return
    }

    set max $::teuk::ddg::max_results
    if {[string equal -nocase $mode "inline"]} { set max $::teuk::ddg::max_inline }

    set res [::teuk::ddg::parse_results $body $max]
    if {[llength $res] == 0} {
        puthelp "PRIVMSG $target :Aucun résultat pour « $q »."
        return
    }

    # COMPACT (default): 8 results, each as "Title — desc URL", separated by orange |, wrapped if needed
    if {[string equal -nocase $mode "compact"]} {
        set want 8
        set parsed [::teuk::ddg::parse_results $body 40]

        if {[llength $parsed] == 0} {
            puthelp "PRIVMSG $target :Aucun résultat pour « $q »."
            return
        }

        set items {}
        foreach r $parsed {
            if {[llength $items] >= $want} { break }

            lassign $r title snip url
            set url  [::teuk::ddg::_normalize_url $url]

            # Drop DDG ad/tracking links and DDG internal redirects/pages
            if {[::teuk::ddg::_is_ddg_ad_url $url]} { continue }
            if {[regexp -nocase {https?://([^/]+)} $url -> host] && [string match -nocase "*duckduckgo.com" $host]} { continue }

            # Prefer short.io if enabled, otherwise keep a compact clickable URL
            set surl [::teuk::ddg::_shorten_url $url]
            if {!$::teuk::ddg::use_shortio} {
                set surl [::teuk::ddg::_compact_url $surl]
            }

            set stitle [::teuk::ddg::_short_title $title 38]

            set chunk ""
            if {$snip ne "-" && $snip ne ""} {
                # dynamic snippet budget to keep chunk reasonable
                set budget [expr {150 - [string length $stitle] - [string length $surl]}]
                if {$budget < 24} { set budget 24 }
                if {$budget > 110} { set budget 110 }
                set ssnip [::teuk::ddg::_short_snippet $snip $budget]
                set chunk "$stitle — $ssnip $::teuk::ddg::color_link\037$surl\037$::teuk::ddg::color_reset"
            } else {
                set chunk "$stitle $::teuk::ddg::color_link\037$surl\037$::teuk::ddg::color_reset"
            }

            lappend items $chunk
        }

        if {[llength $items] == 0} {
            puthelp "PRIVMSG $target :Aucun résultat propre pour « $q »."
            return
        }

        ::teuk::ddg::_send_inline $target $items
        return
    }

    # INLINE: titles + links only (snippets removed to keep clickable URLs)
    if {[string equal -nocase $mode "inline"]} {
        # We parse MORE than 8 then filter (ads/tracking) to still get 8 clean results
        set want $::teuk::ddg::max_inline
        set parsed [::teuk::ddg::parse_results $body 30]

        if {[llength $parsed] == 0} {
            puthelp "PRIVMSG $target :Aucun résultat pour « $q »."
            return
        }

        set items {}
        foreach r $parsed {
            if {[llength $items] >= $want} { break }

            lassign $r title snip url
            set url  [::teuk::ddg::_normalize_url $url]

            # Drop DDG ad/tracking links (the huge duckduckgo.com/y.js?... stuff)
            if {[::teuk::ddg::_is_ddg_ad_url $url]} { continue }

            # Prefer short.io if enabled, otherwise keep a compact clickable URL
            set surl [::teuk::ddg::_shorten_url $url]
            if {!$::teuk::ddg::use_shortio} {
                set surl [::teuk::ddg::_compact_url $surl]
            }

            set stitle [::teuk::ddg::_short_title $title 48]

            # Chunk is short: "Title <url>"
            lappend items "$stitle $::teuk::ddg::color_link\037$surl\037$::teuk::ddg::color_reset"
        }

        if {[llength $items] == 0} {
            puthelp "PRIVMSG $target :Aucun résultat propre pour « $q »."
            return
        }

        # Best effort: one line with orange |, wrap if needed
        ::teuk::ddg::_send_inline $target $items
        return
    }

    # LINES: 4 results, snippet auto-trimmed so URL is always present
    set lines {}
    foreach r $res {
        lassign $r title snip url
        set surl [::teuk::ddg::_shorten_url $url]

        if {$snip ne "-" && $snip ne ""} {
            set snip [::teuk::ddg::_fit_snippet $title $snip $surl 420]
        }

        if {$snip eq "-" || $snip eq ""} {
            set line "$::teuk::ddg::bullet $title $::teuk::ddg::color_link\037$surl\037$::teuk::ddg::color_reset"
        } else {
            set line "$::teuk::ddg::bullet $title — $snip $::teuk::ddg::color_link\037$surl\037$::teuk::ddg::color_reset"
        }
        lappend lines $line
    }

    ::teuk::ddg::_send_lines $target $lines
}

###############################################################################
# BINDS
###############################################################################

proc ::teuk::ddg::cmd_ddg_pub {nick uhost hand chan text} {
    if {![channel get $chan ddg]} {
        ::teuk::ddg::log2 "chan $chan sans +ddg, ignore."
        return
    }

    set q [string trim $text]
    set mode "compact"
    set legacy 0

    if {$q eq ""} {
        puthelp "PRIVMSG $chan :Syntaxe: !ddg <recherche>  |  !ddg -full <recherche>  |  !ddg -full -inline <recherche>"
        return
    }

    if {[regexp -nocase {^-(full)\s+(.+)$} $q -> _ rest]} {
        set legacy 1
        set q [string trim $rest]
        set mode "lines"
    }

    # Legacy behavior: default lines, optional -inline
    if {$legacy && [regexp -nocase {^-(inline)\s+(.+)$} $q -> _ rest2]} {
        set mode "inline"
        set q [string trim $rest2]
    }

    # New default: ignore optional -inline (kept for compatibility)
    if {!$legacy && [regexp -nocase {^-(inline)\s+(.+)$} $q -> _ rest3]} {
        set q [string trim $rest3]
    }

    if {[catch { ::teuk::ddg::do_search $nick $chan $q $mode } e]} {
        ::teuk::ddg::log1 "erreur interne pub: $e"
        puthelp "PRIVMSG $chan :Erreur interne: $e"
    }
}
bind pub - "!ddg" ::teuk::ddg::cmd_ddg_pub

proc ::teuk::ddg::msg_ddg {nick uhost hand text} {
    set q [string trim $text]
    set mode "compact"
    set legacy 0

    if {$q eq ""} {
        puthelp "NOTICE $nick :Syntaxe: ddg <recherche>  |  ddg -full <recherche>  |  ddg -full -inline <recherche>"
        return
    }

    if {[regexp -nocase {^-(full)\s+(.+)$} $q -> _ rest]} {
        set legacy 1
        set q [string trim $rest]
        set mode "lines"
    }

    if {$legacy && [regexp -nocase {^-(inline)\s+(.+)$} $q -> _ rest2]} {
        set mode "inline"
        set q [string trim $rest2]
    }

    if {!$legacy && [regexp -nocase {^-(inline)\s+(.+)$} $q -> _ rest3]} {
        set q [string trim $rest3]
    }

    if {[catch { ::teuk::ddg::do_search $nick $nick $q $mode } e]} {
        ::teuk::ddg::log1 "erreur interne msg: $e"
        puthelp "NOTICE $nick :Erreur interne: $e"
    }
}
bind msg -|- ddg ::teuk::ddg::msg_ddg

###############################################################################
# PARTYLINE
###############################################################################

proc teuk::ddg::setdebug {lvl} {
    if {![string is integer -strict $lvl]} {
        return "usage: .tcl eval teuk::ddg::setdebug <0-3>"
    }
    set ::teuk::ddg::debug $lvl
    return "ddg: debug=$lvl"
}

proc teuk::ddg::setshort {apikey domain} {
    set ::teuk::ddg::short_api_key $apikey
    set ::teuk::ddg::short_domain  $domain
    set ::teuk::ddg::use_shortio   1
    return "ddg: short.io activé (domaine='$domain')"
}

proc teuk::ddg::setshortcheck {onoff} {
    if {![string is integer -strict $onoff]} { return "usage: .tcl eval teuk::ddg::setshortcheck <0|1>" }
    set ::teuk::ddg::short_check $onoff
    return "ddg: short_check=$onoff (0=off par défaut)"
}

###############################################################################
# END
###############################################################################

putlog "ddg: script ddg.tcl v$::teuk::ddg::version chargé (auteur : Te\[u]K). Commande !ddg, chanset +ddg requis."
bind evnt - prerehash ::teuk::ddg::unload
