###############################################################################
# chatGPT.tcl
# v1.3.5 (01/11/2025)  ©2025-2025 Te[u]K  (avec l'aide de MenzAgitat)
# https://teuk.org
###############################################################################
# Public : !chatgpt <prompt>
# Privé  : /msg <BotNick> chatgpt <prompt>
# Aux (si +chatgpt sur chan) :
#   !gptreset
#   !gptset hist <n>
#   !gptset chars <n>
#   !gptecho <user|assistant> <texte>
###############################################################################

if {[info commands ::teuk::chatgpt::unload] eq "::teuk::chatgpt::unload"} { ::teuk::chatgpt::unload }
if { [lindex [split $::version] 1] < 1080404 } { putloglev o * "\00304\[chatgpt - erreur\]\003 Eggdrop trop ancien: \00304${::version}\003 (>= 1.8.4 requis). Chargement annulé."; return }
if { [catch { package require Tcl 8.6 }] } { putloglev o * "\00304\[chatgpt - erreur\]\003 Tcl 8.6+ requis. Version actuelle: \00304${::tcl_version}\003. Chargement annulé."; return }
if { [catch { package require tls } terr] } { putloglev o * "\00304\[chatgpt - erreur\]\003 package Tcl 'tls' requis pour HTTPS (tcl-tls). Erreur: \00304$terr\003. Chargement annulé."; return }

package provide chatgpt 1.3.5

namespace eval ::teuk::chatgpt {
    # ---- conf ----
    variable version "1.3.5"
    variable debug  2

    variable api_key "YOUR API KEY HERE"
    variable api_url "https://api.openai.com/v1/chat/completions"
    variable model "gpt-4o-mini"
    variable temperature 0.5
    variable max_tokens 400

    variable use_wrapper_https 1
    variable http_decode_utf8  1

    variable max_privmsg 6
    variable trunc_msg  " ...<truncated>"
    variable sleep_ms   600

    variable hist_max_msgs  10
    variable hist_max_chars 4000

    # ---- runtime ----
    variable HAVE_TLS 0
    variable HAVE_JSON 0
    variable hist;           array set hist {}
    variable hist_overrides; array set hist_overrides {}

    catch { setudef flag chatgpt }
}

proc ::teuk::chatgpt::log {lvl tag msg} {
    if {$::teuk::chatgpt::debug >= $lvl} { putlog "chatgpt v$::teuk::chatgpt::version $tag: $msg" }
}
proc ::teuk::chatgpt::log1 {msg} { ::teuk::chatgpt::log 1 INFO  $msg }
proc ::teuk::chatgpt::log2 {msg} { ::teuk::chatgpt::log 2 DBG   $msg }
proc ::teuk::chatgpt::log3 {msg} { ::teuk::chatgpt::log 3 TRACE $msg }
proc ::teuk::chatgpt::_ms {} { clock milliseconds }

# --- deps ---
if {[catch {package require http} e]} {
    putlog "chatgpt: ERREUR package http requis ($e)"
    return
}
::teuk::chatgpt::log1 "http pkg: [package provide http]"

if {![catch {package require json} e]} {
    set ::teuk::chatgpt::HAVE_JSON 1
    ::teuk::chatgpt::log1 "json pkg: [package provide json]"
} else {
    ::teuk::chatgpt::log1 "json pkg: ABSENT (parse regex fallback)"
}

if {[catch {package require tls} e]} {
    set ::teuk::chatgpt::HAVE_TLS 0
    putlog "chatgpt: WARNING: package tls absent (HTTPS requis)"
    ::teuk::chatgpt::log1 "tls pkg: ABSENT ($e)"
} else {
    set ::teuk::chatgpt::HAVE_TLS 1
    ::teuk::chatgpt::log1 "tls pkg: [package provide tls]"
    catch { ::tls::init -autoservername 1 -tls1.3 1 -tls1.2 1 }
}

catch { ::http::config -useragent "Eggdrop-ChatGPT/1.0 (+tcl)" }
::teuk::chatgpt::log2 "http user-agent configuré"

# --- unload propre à la Menz ---
proc ::teuk::chatgpt::unload {args} {
    set ns [namespace current]
    putlog "chatgpt: nettoyage du namespace $ns..."
    foreach b [binds *] {
        if {[llength $b] < 5} { continue }
        set cmd [lindex $b 4]
        if {[string match "${ns}::*" $cmd]} {
            catch { unbind [lindex $b 0] [lindex $b 1] [lindex $b 2] $cmd }
        }
    }
    catch { ::http::unregister httpsgpt }
    catch { unset -nocomplain ::teuk::chatgpt::hist }
    catch { unset -nocomplain ::teuk::chatgpt::hist_overrides }
    catch { namespace delete $ns }
}

# --- TLS wrapper isolé : httpsgpt:// ---
proc ::teuk::chatgpt::_tls_socket {args} {
    set async 0; set host ""; set port ""; set myaddr ""; set myport ""; set keepalive ""
    set i 0; set n [llength $args]
    while {$i < $n} {
        set a [lindex $args $i]
        if {[string match -* $a]} {
            switch -- $a {
                -async     { set async 1 }
                -myaddr    { incr i; set myaddr  [lindex $args $i] }
                -myport    { incr i; set myport  [lindex $args $i] }
                -keepalive { set keepalive 1 }
                default {}
            }
        } elseif {$host eq ""} { set host $a } elseif {$port eq ""} { set port $a }
        incr i
    }
    if {$host eq "" || $port eq ""} { error "_tls_socket: args invalides: $args" }

    set opts {}
    if {$myaddr ne ""}    { lappend opts -myaddr $myaddr }
    if {$myport ne ""}    { lappend opts -myport $myport }
    if {$keepalive ne ""} { lappend opts -keepalive $keepalive }

    set s [::socket {*}$opts $host $port]
    catch { fconfigure $s -translation binary -encoding binary -buffering none }
    if {$async} { catch { fconfigure $s -blocking 0 } }

    set imported 0
    foreach attempt {
        {-autoservername 1 -servername __HOST__}
        {-servername __HOST__}
        {}
    } {
        set cmd [list ::tls::import $s]
        if {[llength $attempt]} {
            foreach {k v} $attempt {
                if {$v eq "__HOST__"} { set v $host }
                lappend cmd $k $v
            }
        }
        if {![catch { {*}$cmd }]} { set imported 1; break }
    }
    if {!$imported} { catch {close $s}; error "tls::import a échoué pour $host:$port" }
    return $s
}
if {$::teuk::chatgpt::HAVE_TLS} { catch { ::http::register httpsgpt 443 ::teuk::chatgpt::_tls_socket } }

proc ::teuk::chatgpt::_rewrite_url {url} {
    if {$::teuk::chatgpt::HAVE_TLS && $::teuk::chatgpt::use_wrapper_https && [string match -nocase "https://*" $url]} {
        return [string map {"https://" "httpsgpt://"} $url]
    }
    return $url
}

# --- JSON helpers ---
proc ::teuk::chatgpt::_json_escape {s} {
    return [string map [list "\\" "\\\\" "\"" "\\\"" "\r" "\\r" "\n" "\\n" "\t" "\\t"] $s]
}
proc ::teuk::chatgpt::_json_str {s} { return "\"[::teuk::chatgpt::_json_escape $s]\"" }

# --- Historique (fenêtre glissante) ---
proc ::teuk::chatgpt::_limits {target} {
    set maxm $::teuk::chatgpt::hist_max_msgs
    set maxc $::teuk::chatgpt::hist_max_chars
    if {[info exists ::teuk::chatgpt::hist_overrides($target)]} {
        set o $::teuk::chatgpt::hist_overrides($target)
        if {[dict exists $o max_msgs]}  { set maxm [dict get $o max_msgs] }
        if {[dict exists $o max_chars]} { set maxc [dict get $o max_chars] }
    }
    return [list $maxm $maxc]
}
proc ::teuk::chatgpt::_trim_history {target} {
    if {![info exists ::teuk::chatgpt::hist($target)]} { return }
    lassign [::teuk::chatgpt::_limits $target] max_msgs max_chars
    set L $::teuk::chatgpt::hist($target)

    if {[llength $L] > $max_msgs} { set L [lrange $L end-[expr {$max_msgs-1}] end] }

    set tot 0; set out {}
    for {set i [expr {[llength $L]-1}]} {$i >= 0} {incr i -1} {
        set m [lindex $L $i]
        set c [dict get $m content]
        incr tot [string length $c]
        if {$tot > $max_chars} { break }
        set out [linsert $out 0 $m]
    }
    set ::teuk::chatgpt::hist($target) $out
}
proc ::teuk::chatgpt::_hist_add {target role content} {
    set msg [dict create role $role content [string trim $content]]
    if {![info exists ::teuk::chatgpt::hist($target)]} {
        set ::teuk::chatgpt::hist($target) [list $msg]
    } else {
        lappend ::teuk::chatgpt::hist($target) $msg
    }
    ::teuk::chatgpt::_trim_history $target
}

# --- Compose messages (IMPORTANT: le prompt courant est déjà dans l'historique) ---
proc ::teuk::chatgpt::_compose_messages {target} {
    set sys "You answer helpfully and precisely. Reply in French. Avoid emojis. Keep it concise."
    set out [list [dict create role system content $sys]]
    if {[info exists ::teuk::chatgpt::hist($target)]} {
        foreach msg $::teuk::chatgpt::hist($target) {
            if {![dict exists $msg role] || ![dict exists $msg content]} { continue }
            lappend out $msg
        }
    }
    return $out
}

# --- Payload JSON (manuel compact, fiable) ---
proc ::teuk::chatgpt::_make_payload {target} {
    set msgs [::teuk::chatgpt::_compose_messages $target]
    set parts {}
    foreach m $msgs {
        set r [dict get $m role]
        set c [dict get $m content]
        lappend parts "{\"role\":[::teuk::chatgpt::_json_str $r],\"content\":[::teuk::chatgpt::_json_str $c]}"
    }
    set jmsgs "\[[join $parts ,]\]"
    return "{\"model\":[::teuk::chatgpt::_json_str $::teuk::chatgpt::model],\"temperature\":$::teuk::chatgpt::temperature,\"max_tokens\":$::teuk::chatgpt::max_tokens,\"messages\":$jmsgs}"
}

# --- Call API ---
proc ::teuk::chatgpt::_call_api {payload} {
    variable api_key
    if {!$::teuk::chatgpt::HAVE_TLS} {
        return -code error "TLS non disponible (installe tcl-tls)"
    }

    set key [string trim $api_key]
    if {$key eq ""} {
        return -code error "cle API manquante (configure ::teuk::chatgpt::api_key)"
    }
    if {[string length $key] < 32} {
        ::teuk::chatgpt::log1 "cle API semble très courte; vérifie ta configuration."
    }
    if {![regexp {^sk-} $key]} {
        ::teuk::chatgpt::log1 "format de cle API inattendu (ne commence pas par 'sk-'); on tente quand même."
    }

    set url [::teuk::chatgpt::_rewrite_url $::teuk::chatgpt::api_url]
    set headers [list Authorization "Bearer $key" Content-Type "application/json; charset=utf-8" Accept "application/json"]
    set body [encoding convertto utf-8 $payload]

    set t0 [::teuk::chatgpt::_ms]
    ::teuk::chatgpt::log2 "API POST $url"
    if {[catch {
        set tok [::http::geturl $url -method POST -headers $headers -type "application/json; charset=utf-8" -query $body -timeout 30000 -strict 1 -binary 1]
    } e]} {
        return -code error "http geturl failed: $e"
    }

    set stat [::http::status $tok]
    set code [::http::ncode  $tok]
    set raw  [::http::data   $tok]
    ::http::cleanup $tok
    ::teuk::chatgpt::log2 "API status=$stat code=$code dt=[expr {[::teuk::chatgpt::_ms]-$t0}]ms"

    if {$::teuk::chatgpt::http_decode_utf8} {
        catch { set raw [encoding convertfrom utf-8 $raw] }
    }
    if {$stat ne "ok"} { return -code error "http $stat (code $code)" }
    if {$code < 200 || $code >= 300} { return -code error "http $code: $raw" }
    return $raw
}

# --- Parse answer ---
proc ::teuk::chatgpt::_parse_answer {response} {
    if {$::teuk::chatgpt::HAVE_JSON} {
        set d [json::json2dict $response]
        set first [lindex [dict get $d choices] 0]
        return [string trim [dict get [dict get $first message] content]]
    }
    if {[regexp {\"content\"\s*:\s*\"((?:\\.|[^\"\\])*)\"} $response -> m]} {
        set m [string map [list "\\n" "\n" "\\r" "\r" "\\t" "\t" "\\\\" "\\" "\\\"" "\""] $m]
        return [string trim $m]
    }
    return -code error "invalid JSON response"
}

# --- IRC wrap/send (maxi chars par ligne, wrap mots, pas de retours inutiles) ---
proc ::teuk::chatgpt::_irc_max_line {target} { return 430 }

proc ::teuk::chatgpt::_wrap {target text} {
    set max [::teuk::chatgpt::_irc_max_line $target]
    set text [string trim $text]
    set out {}
    while {[string length $text] > 0} {
        if {[string length $text] <= $max} { lappend out $text; break }
        set slice [string range $text 0 $max]
        set cut [string last " " $slice]
        if {$cut < 0} { set cut $max }
        lappend out [string trimright [string range $text 0 $cut]]
        set text [string trimleft [string range $text [expr {$cut+1}] end]]
    }
    return $out
}

proc ::teuk::chatgpt::_send {target chunks} {
    set L $chunks
    if {[llength $L] > $::teuk::chatgpt::max_privmsg} {
        set last [expr {$::teuk::chatgpt::max_privmsg - 1}]
        set c [lindex $L $last]
        set allow [expr {[string length $c] - [string length $::teuk::chatgpt::trunc_msg]}]
        if {$allow < 20} { set allow 20 }
        set c [string range $c 0 $allow]
        set cut [string last " " $c]
        if {$cut > 0} { set c [string range $c 0 $cut] }
        set L [lreplace $L $last end "[string trimright $c]$::teuk::chatgpt::trunc_msg"]
    }
    foreach line $L {
        if {$line eq ""} { continue }
        puthelp "PRIVMSG $target :$line"
        after $::teuk::chatgpt::sleep_ms
    }
}

# --- Pipeline ---
proc ::teuk::chatgpt::_handle {nick target prompt} {
    set prompt [regsub -all {\s+} [string trim $prompt] " "]
    if {$prompt eq ""} {
        puthelp "NOTICE $nick :Syntaxe: !chatgpt <prompt>"
        return
    }

    ::teuk::chatgpt::_hist_add $target user $prompt

    if {[catch { set payload [::teuk::chatgpt::_make_payload $target] } e]} {
        ::teuk::chatgpt::log1 "payload error: $e"
        puthelp "NOTICE $nick :Erreur interne (payload)."
        return
    }

    if {[catch { set resp [::teuk::chatgpt::_call_api $payload] } e]} {
        ::teuk::chatgpt::log1 "API error: $e"
        puthelp "NOTICE $nick :API indisponible."
        return
    }

    if {[catch { set ans [::teuk::chatgpt::_parse_answer $resp] } e]} {
        ::teuk::chatgpt::log1 "parse error: $e"
        puthelp "NOTICE $nick :Réponse API illisible."
        return
    }

    set ans [regsub -all {[\r\n]+} $ans " "]
    set ans [regsub -all {\s{2,}}  $ans " "]
    set ans [string trim $ans]

    ::teuk::chatgpt::_hist_add $target assistant $ans
    ::teuk::chatgpt::_send $target [::teuk::chatgpt::_wrap $target $ans]
}

# --- Binds : compact (un seul bind pubm pour chat + aux) ---
proc ::teuk::chatgpt::pubm {nick uhost hand chan text} {
    if {![channel get $chan chatgpt]} { return }

    if {[regexp -nocase -- {^!chatgpt\s+(.+)$} $text -> q]} {
        ::teuk::chatgpt::_handle $nick $chan $q
        return
    }

    if {[regexp -nocase -- {^!gptreset\s*$} $text]} {
        unset -nocomplain ::teuk::chatgpt::hist($chan)
        puthelp "PRIVMSG $chan :OK historique reinitialise pour $chan"
        return
    }

    if {[regexp -nocase -- {^!gptset\s+hist\s+(\d+)\s*$} $text -> n]} {
        set cur [dict create]
        if {[info exists ::teuk::chatgpt::hist_overrides($chan)]} { set cur $::teuk::chatgpt::hist_overrides($chan) }
        set ::teuk::chatgpt::hist_overrides($chan) [dict merge $cur [dict create max_msgs $n]]
        puthelp "PRIVMSG $chan :OK hist_max_msgs=$n pour $chan"
        return
    }

    if {[regexp -nocase -- {^!gptset\s+chars\s+(\d+)\s*$} $text -> n]} {
        set cur [dict create]
        if {[info exists ::teuk::chatgpt::hist_overrides($chan)]} { set cur $::teuk::chatgpt::hist_overrides($chan) }
        set ::teuk::chatgpt::hist_overrides($chan) [dict merge $cur [dict create max_chars $n]]
        puthelp "PRIVMSG $chan :OK hist_max_chars=$n pour $chan"
        return
    }

    if {[regexp -nocase -- {^!gptecho\s+(user|assistant)\s+(.+)$} $text -> role payload]} {
        ::teuk::chatgpt::_hist_add $chan $role $payload
        puthelp "PRIVMSG $chan :INFO ajoute au contexte ($role): [string range $payload 0 80]"
        return
    }
}
bind pubm -|- * ::teuk::chatgpt::pubm

proc ::teuk::chatgpt::msg_chatgpt {nick uhost hand text} {
    if {$text eq ""} {
        puthelp "NOTICE $nick :Syntaxe: chatgpt <prompt>"
        return
    }
    ::teuk::chatgpt::_handle $nick $nick $text
}
bind msg -|- chatgpt ::teuk::chatgpt::msg_chatgpt

proc ::teuk::chatgpt::msg_aux {nick uhost hand text} {
    if {[regexp -nocase -- {^gptreset\s*$} $text]} {
        unset -nocomplain ::teuk::chatgpt::hist($nick)
        puthelp "NOTICE $nick :Historique reinitialise pour $nick"
        return
    }
    if {[regexp -nocase -- {^gptset\s+hist\s+(\d+)\s*$} $text -> n]} {
        set cur [dict create]
        if {[info exists ::teuk::chatgpt::hist_overrides($nick)]} { set cur $::teuk::chatgpt::hist_overrides($nick) }
        set ::teuk::chatgpt::hist_overrides($nick) [dict merge $cur [dict create max_msgs $n]]
        puthelp "NOTICE $nick :hist_max_msgs=$n pour $nick"
        return
    }
    if {[regexp -nocase -- {^gptset\s+chars\s+(\d+)\s*$} $text -> n]} {
        set cur [dict create]
        if {[info exists ::teuk::chatgpt::hist_overrides($nick)]} { set cur $::teuk::chatgpt::hist_overrides($nick) }
        set ::teuk::chatgpt::hist_overrides($nick) [dict merge $cur [dict create max_chars $n]]
        puthelp "NOTICE $nick :hist_max_chars=$n pour $nick"
        return
    }
    if {[regexp -nocase -- {^gptecho\s+(user|assistant)\s+(.+)$} $text -> role payload]} {
        ::teuk::chatgpt::_hist_add $nick $role $payload
        puthelp "NOTICE $nick :ajout contexte ($role): [string range $payload 0 80]"
        return
    }
}
bind msg -|- gptreset ::teuk::chatgpt::msg_aux
bind msg -|- gptset   ::teuk::chatgpt::msg_aux
bind msg -|- gptecho  ::teuk::chatgpt::msg_aux

# partyline helpers (facultatif mais compact)
proc teuk::chatgpt::setkey {k} { set ::teuk::chatgpt::api_key $k }
proc teuk::chatgpt::setmode {m} {
    if {!$::teuk::chatgpt::HAVE_TLS} { return "TLS indisponible" }
    set m [string tolower [string trim $m]]
    if {$m ni {wrapper direct}} { return "usage: .tcl eval teuk::chatgpt::setmode wrapper|direct" }
    set ::teuk::chatgpt::use_wrapper_https [expr {$m eq "wrapper"}]
    return "OK mode=$m"
}

putlog "chatgpt: script chatGPT v$::teuk::chatgpt::version charge (auteur: Te\[u]K, aide: MenzAgitat). Commande !chatgpt, chanset +chatgpt requis."
bind evnt - prerehash ::teuk::chatgpt::unload
