# chatGPT.tcl — Eggdrop + OpenAI (Tcl 8.6+, http+tls, json optionnel)
# Public: !chatgpt <prompt>  (chanset +chatgpt)
# Privé : /msg <BotNick> chatgpt <prompt>
# Historique: mémorisation par canal/privé avec fenêtre glissante.

# 0) Version & Debug
namespace eval ::teuk::chatgpt {
    variable script_version "1.3.2"
    variable debug 2  ;# 0=min, 1=normal, 2=verbeux, 3=très verbeux
}
proc ::teuk::chatgpt::dlog {lvl msg} {
    if {$lvl <= $::teuk::chatgpt::debug} {
        putlog "chatgpt v$::teuk::chatgpt::script_version d$lvl: $msg"
    }
}

# 1) Namespace & mémoire
namespace eval ::teuk::chatgpt {
    # Historique: dict target -> list {role user|assistant content "text"}
    variable hist
    array set hist {}
    # Paramètres d'historique (fenêtre globale par défaut)
    variable hist_max_msgs 10       ;# nombre de messages (user/assistant comptés individuellement)
    variable hist_max_chars 4000    ;# budget approx en caractères (avant le prompt courant)
    # Overrides par canal: dict chan -> {max_msgs N max_chars M}
    variable hist_overrides
    array set hist_overrides {}
}

# 2) Packages
if {[catch {package require http} perr]} { error "chatgpt: package http requis: $perr" }
::teuk::chatgpt::dlog 1 "http pkg: [package provide http]"

# JSON — json::write si dispo; sinon fallback manuel
set ::teuk::chatgpt::HAVE_JSON 0
set ::teuk::chatgpt::HAVE_JSON_WRITE 0
set ::teuk::chatgpt::JSON_MODE "manual"
if {![catch {package require json} jerr]} {
    set jver [package provide json]
    if {[llength [info commands ::json::write]] > 0} {
        set ::teuk::chatgpt::HAVE_JSON 1
        set ::teuk::chatgpt::HAVE_JSON_WRITE 1
        set ::teuk::chatgpt::JSON_MODE "write"
        ::teuk::chatgpt::dlog 1 "json pkg: $jver (mode=write)"
    } else {
        ::teuk::chatgpt::dlog 1 "json pkg: $jver (json::write absent) — fallback manuel"
    }
} else {
    ::teuk::chatgpt::dlog 1 "json pkg: ABSENT — fallback manuel"
}

# TLS
if {[catch {package require tls} terr]} {
    putlog "chatgpt: WARNING: package tls absent (HTTPS requis)"
    set ::teuk::chatgpt::HAVE_TLS 0
    ::teuk::chatgpt::dlog 1 "tls pkg: ABSENT ($terr)"
} else {
    set ::teuk::chatgpt::HAVE_TLS 1
    set tver [package provide tls]
    ::teuk::chatgpt::dlog 1 "tls pkg: $tver"
    catch { ::tls::init -autoservername 1 -tls1.3 1 -tls1.2 1 }
    ::teuk::chatgpt::dlog 2 "tls::init appliqué (best effort)"
}

# User-Agent HTTP
catch { ::http::config -useragent "Eggdrop-ChatGPT/1.0 (+tcl)" }
::teuk::chatgpt::dlog 2 "http user-agent configuré"

# 2bis) Options runtime IRC
namespace eval ::teuk::chatgpt {
    variable use_wrapper_https 1       ;# 1 => utilise le schéma httpsgpt:// (wrapper TLS maison)
    variable http_decode_utf8 1
    variable send_mode unicode

    variable wrap_bytes   420
    variable max_privmsg  6
    variable trunc_msg    " ...<truncated>"
    variable sleep_ms     600
}
proc ::teuk::chatgpt::_ms {} { return [clock milliseconds] }

# 3) Config API
namespace eval ::teuk::chatgpt {
    variable api_key      "YOUR_API_KEY_HERE"
    variable api_url      "https://api.openai.com/v1/chat/completions"
    variable model        "gpt-4o-mini"
    variable temperature  0.5
    variable max_tokens   400
    catch { setudef flag chatgpt }
}

# 4) HTTPS SNI wrapper (robuste Debian 9/12)
proc ::teuk::chatgpt::_tls_socket {args} {
    # Accepte: -async [0|1], -myaddr <ip>, -myport <port>, -keepalive [val], puis host port.
    set async 0
    set myaddr ""; set myport ""; set keepalive ""
    set host ""; set port ""

    set i 0
    set n [llength $args]
    while {$i < $n} {
        set a [lindex $args $i]
        if {[string match -* $a]} {
            switch -- $a {
                -async {
                    set async 1
                    if {$i+1 < $n} {
                        set nxt [lindex $args [expr {$i+1}]]
                        if {[string is integer -strict $nxt]} {
                            set async [expr {$nxt ? 1 : 0}]
                            incr i
                        }
                    }
                }
                -myaddr { if {$i+1 < $n} { incr i; set myaddr [lindex $args $i] } }
                -myport { if {$i+1 < $n} { incr i; set myport [lindex $args $i] } }
                -keepalive {
                    set keepalive 1
                    if {$i+1 < $n} {
                        set nxt [lindex $args [expr {$i+1}]]
                        if {[string is integer -strict $nxt]} {
                            set keepalive $nxt
                            incr i
                        }
                    }
                }
                default { }
            }
        } elseif {$host eq ""} {
            set host $a
        } elseif {$port eq ""} {
            set port $a
        }
        incr i
    }
    if {$host eq "" || $port eq ""} {
        ::teuk::chatgpt::dlog 1 "_tls_socket args invalides: $args"
        error "_tls_socket: args invalides"
    }
    ::teuk::chatgpt::dlog 2 "_tls_socket(host=$host port=$port async=$async myaddr=$myaddr myport=$myport)"

    # socket TCP
    set sockopts {}
    if {$myaddr ne ""}   { lappend sockopts -myaddr $myaddr }
    if {$myport ne ""}   { lappend sockopts -myport $myport }
    if {$keepalive ne "" && $keepalive != ""} { lappend sockopts -keepalive $keepalive }
    if {[catch { set s [::socket {*}$sockopts $host $port] } e]} {
        ::teuk::chatgpt::dlog 1 "socket error $host:$port -> $e"
        return -code error $e
    }
    if {$async} { catch { fconfigure $s -blocking 0 } } else { catch { fconfigure $s -blocking 1 } }
    catch { fconfigure $s -translation binary -encoding binary -buffering none }

    # Upgrade TLS via tls::import (SNI si possible)
    set imported 0
    set errlist {}
    foreach attempt {
        {-autoservername 1 -servername __HOST__}
        {-servername __HOST__}
        {}
    } {
        set cmd [list ::tls::import $s]
        if {[llength $attempt]} {
            foreach {k v} $attempt {
                if {$v eq "__HOST__"} { set v $host }
                lappend cmd $k $v
            }
        }
        if {[catch { {*}$cmd } e]} {
            lappend errlist $e
            continue
        } else { set imported 1; break }
    }
    if {!$imported} {
        catch { close $s }
        ::teuk::chatgpt::dlog 1 "tls::import failed: [join $errlist { | }]"
        return -code error "tls::import failed"
    }
    return $s
}

# 🔒 IMPORTANT: on NE TOUCHE PAS au handler https global.
# On enregistre un schéma dédié "httpsgpt" pour notre wrapper → aucune interférence.
if {$::teuk::chatgpt::HAVE_TLS} {
    catch { ::http::register httpsgpt 443 ::teuk::chatgpt::_tls_socket }
    ::teuk::chatgpt::dlog 2 "http::register httpsgpt -> wrapper OK (isolé)"
}

# 5) Helpers JSON + Historique
proc ::teuk::chatgpt::_json_escape {s} {
    set mapping [list "\\" "\\\\" "\"" "\\\"" "\r" "\\r" "\n" "\\n" "\t" "\\t"]
    return [string map $mapping $s]
}

# Retourne liste de messages {role content} prête pour l'API (sys + hist + user)
proc ::teuk::chatgpt::_compose_messages {target prompt} {
    # Système (ASCII-only)
    set sys "You answer helpfully and precisely. In French, never start with 'Oh la la'. Reply in max 10 lines, line-based. No emojis."
    set out [list [list role system content $sys]]

    # Historique pour target (si existe)
    if {[info exists ::teuk::chatgpt::hist($target)]} {
        foreach msg $::teuk::chatgpt::hist($target) {
            if {![dict exists $msg role] || ![dict exists $msg content]} { continue }
            lappend out [list role [dict get $msg role] content [dict get $msg content]]
        }
    }
    # Prompt courant (user)
    lappend out [list role user content [string trim $prompt]]
    return $out
}

# Convertit une liste { {role x content y} ... } en JSON selon mode
proc ::teuk::chatgpt::_messages_to_json {msgs} {
    if {$::teuk::chatgpt::JSON_MODE eq "write"} {
        set arr {}
        foreach m $msgs {
            set r [lindex $m [lsearch -exact $m role]+1]
            set c [lindex $m [lsearch -exact $m content]+1]
            set jr [json::write string $r]
            set jc [json::write string $c]
            lappend arr [json::write object role $jr content $jc]
        }
        return [json::write array {*}$arr]
    } else {
        set parts {}
        foreach m $msgs {
            set r [lindex $m [lsearch -exact $m role]+1]
            set c [lindex $m [lsearch -exact $m content]+1]
            set jr [::teuk::chatgpt::_json_escape $r]
            set jc [::teuk::chatgpt::_json_escape $c]
            lappend parts "{\"role\":\"$jr\",\"content\":\"$jc\"}"
        }
        return "\[[join $parts ,]\]"
    }
}

# Trim l'historique d'un target selon limites (fenêtre glissante)
proc ::teuk::chatgpt::_trim_history {target} {
    variable hist_max_msgs
    variable hist_max_chars
    set max_msgs $hist_max_msgs
    set max_chars $hist_max_chars
    if {[info exists ::teuk::chatgpt::hist_overrides($target)]} {
        set o $::teuk::chatgpt::hist_overrides($target)
        if {[dict exists $o max_msgs]}  { set max_msgs  [dict get $o max_msgs] }
        if {[dict exists $o max_chars]} { set max_chars [dict get $o max_chars] }
    }
    if {![info exists ::teuk::chatgpt::hist($target)]} { return }

    set L $::teuk::chatgpt::hist($target)
    if {[llength $L] > $max_msgs} {
        set L [lrange $L end-[expr {$max_msgs-1}] end]
    }
    set tot 0
    set out {}
    for {set i [expr {[llength $L]-1}]} {$i >= 0} {incr i -1} {
        set m [lindex $L $i]
        set c [dict get $m content]
        incr tot [string length $c]
        if {$tot > $max_chars} { break }
        set out [linsert $out 0 $m]
    }
    set ::teuk::chatgpt::hist($target) $out
}

# Ajoute un message à l'historique (role=user|assistant)
proc ::teuk::chatgpt::_hist_add {target role content} {
    set msg [dict create role $role content [string trim $content]]
    if {![info exists ::teuk::chatgpt::hist($target)]} {
        set ::teuk::chatgpt::hist($target) [list $msg]
    } else {
        lappend ::teuk::chatgpt::hist($target) $msg
    }
    ::teuk::chatgpt::_trim_history $target
}

# 6) Payload & API
proc ::teuk::chatgpt::_make_payload {target prompt} {
    variable model; variable temperature; variable max_tokens
    set msgs [::teuk::chatgpt::_compose_messages $target $prompt]
    set jmessages [::teuk::chatgpt::_messages_to_json $msgs]

    if {$::teuk::chatgpt::JSON_MODE eq "write"} {
        set jmodel [json::write string $model]
        return [json::write object \
            model $jmodel \
            temperature $temperature \
            max_tokens $max_tokens \
            messages $jmessages \
        ]
    } else {
        set template {{"model":"%MODEL%","temperature":%TEMP%,"max_tokens":%TOK%,"messages":%MSGS%}}
        set payload [string map [list %MODEL% $model %TEMP% $temperature %TOK% $max_tokens %MSGS% $jmessages] $template]
        ::teuk::chatgpt::dlog 2 "payload(json-manual) => [string range $payload 0 300]..."
        return $payload
    }
}

proc ::teuk::chatgpt::_call_api {payload} {
    variable api_key; variable api_url
    variable http_decode_utf8
    variable use_wrapper_https

    if {$api_key eq ""} { return -code error "cle API manquante" }
    if {!$::teuk::chatgpt::HAVE_TLS} { return -code error "TLS non disponible (installe tcl-tls)" }

    # Utilise un schéma isolé "httpsgpt://" pour nos requêtes si wrapper activé
    set url $api_url
    if {$use_wrapper_https} {
        if {[string match -nocase "https://*" $api_url]} {
            set url [string map {"https://" "httpsgpt://"} $api_url]
        }
    }

    set headers [list Authorization "Bearer $api_key" Content-Type "application/json; charset=utf-8" Accept "application/json"]
    set body    [encoding convertto utf-8 $payload]

    set t0 [::teuk::chatgpt::_ms]
    ::teuk::chatgpt::dlog 2 "http::geturl -> $url (POST, timeout=30000, strict=1, binary=1)"
    if {[catch {
        set tok [::http::geturl $url -method POST -headers $headers -type "application/json; charset=utf-8" -query $body -timeout 30000 -strict 1 -binary 1]
    } err]} {
        set dt [expr {[::teuk::chatgpt::_ms]-$t0}]
        ::teuk::chatgpt::dlog 1 "http geturl error -> $err (dt=${dt}ms)"
        return -code error "http geturl failed: $err"
    }

    set stat [::http::status $tok]
    set code [::http::ncode  $tok]
    set dt [expr {[::teuk::chatgpt::_ms]-$t0}]
    ::teuk::chatgpt::dlog 2 "http status=$stat ncode=$code dt=${dt}ms"

    set raw [::http::data $tok]
    ::http::cleanup $tok

    if {$http_decode_utf8} {
        catch { set raw [encoding convertfrom utf-8 $raw] }
    }
    if {$stat ne "ok"} { return -code error "http $stat (code $code)" }
    if {$code < 200 || $code >= 300} { return -code error "http $code: $raw" }
    return $raw
}

# 7) Parsing JSON
proc ::teuk::chatgpt::_parse_answer {response} {
    if {$::teuk::chatgpt::HAVE_JSON && $::teuk::chatgpt::HAVE_JSON_WRITE} {
        set d [json::json2dict $response]
        if {![dict exists $d choices]} { return -code error "invalid JSON: no choices" }
        set choices [dict get $d choices]
        if {[llength $choices] < 1} { return -code error "invalid JSON: empty choices" }
        set first [lindex $choices 0]
        if {![dict exists $first message]} { return -code error "invalid JSON: no message" }
        set msg [dict get $first message]
        if {![dict exists $msg content]} { return -code error "invalid JSON: no content" }
        set content [dict get $msg content]
        return [string trim $content]
    } else {
        if {[regexp {\"content\"\s*:\s*\"((?:\\.|[^\"\\])*)\"} $response -> m]} {
            set mapping [list "\\n" "\n" "\\r" "\r" "\\t" "\t" "\\\\" "\\" "\\\"" "\""]
            set m [string map $mapping $m]
            return [string trim $m]
        }
        return -code error "invalid JSON response"
    }
}

# 8) Wrap + Envoi IRC
proc ::teuk::chatgpt::_wrap_chunks {txt} {
    variable wrap_bytes
    set out {}
    set txt [string trim $txt]
    while {[string length $txt] > 0} {
        if {[string length $txt] <= $wrap_bytes} { lappend out $txt ; break }
        set slice [string range $txt 0 [expr {$wrap_bytes-1}]]
        set brk [string last " " $slice]
        if {$brk < 0} { set brk [expr {$wrap_bytes-1}] }
        lappend out [string trimright [string range $txt 0 $brk]]
        set txt [string trimleft [string range $txt [expr {$brk+1}] end]]
    }
    return $out
}
proc ::teuk::chatgpt::_send_chunks {target chunks} {
    variable max_privmsg; variable trunc_msg; variable sleep_ms
    variable send_mode
    set send_mode unicode

    set count [llength $chunks]
    set last [expr {$count-1}]
    if {$count > $max_privmsg} {
        set last [expr {$max_privmsg-1}]
        set lastChunk [lindex $chunks $last]
        set allow [expr {[string length $lastChunk] - [string length $trunc_msg]}]
        if {$allow < 1} { set allow 1 }
        set chopped [string range $lastChunk 0 $allow]
        set space [string last " " $chopped]
        if {$space > 0} { set chopped [string range $chopped 0 $space] }
        set chopped [string trimright $chopped]
        set chunks [lreplace $chunks $last $last "${chopped}${trunc_msg}"]
    }
    for {set i 0} {$i <= $last} {incr i} {
        set line [lindex $chunks $i]
        if {$line eq ""} { continue }
        set msg "PRIVMSG $target :$line"
        if {$::teuk::chatgpt::debug >= 3} {
            set want [encoding convertto utf-8 $msg]
            set wanthex [binary encode hex [string range $want 0 63]]
            ::teuk::chatgpt::dlog 3 "SEND(unicode) HEX\[0..63]=$wanthex"
        }
        puthelp $msg
        after $sleep_ms
    }
    ::teuk::chatgpt::dlog 1 "sent [expr {$last+1}] PRIVMSG to $target (mode=unicode)"
}

# 9) Pipeline avec Historique
proc ::teuk::chatgpt::_handle_prompt {nick target prompt} {
    ::teuk::chatgpt::dlog 3 "prompt from $nick (hidden)"
    set prompt [regsub -all {\s+} [string trim $prompt] " "]
    if {$prompt eq ""} { puthelp "NOTICE $nick :Syntaxe: !chatgpt <prompt>" ; return }

    # Ajout à l'historique (role=user)
    ::teuk::chatgpt::_hist_add $target user $prompt

    # Payload avec historique + user
    if {[catch {set payload [::teuk::chatgpt::_make_payload $target $prompt]} perr]} {
        ::teuk::chatgpt::dlog 1 "payload error: $perr"
        puthelp "NOTICE $nick :Erreur interne (payload)"
        # rollback user
        if {[info exists ::teuk::chatgpt::hist($target)]} {
            set L $::teuk::chatgpt::hist($target)
            if {[llength $L]>0 && [dict get [lindex $L end] role] eq "user"} {
                set ::teuk::chatgpt::hist($target) [lrange $L 0 end-1]
            }
        }
        return
    }

    if {[catch {set resp [::teuk::chatgpt::_call_api $payload]} herr]} {
        ::teuk::chatgpt::dlog 1 $herr
        puthelp "NOTICE $nick :API indisponible."
        # rollback user
        if {[info exists ::teuk::chatgpt::hist($target)]} {
            set L $::teuk::chatgpt::hist($target)
            if {[llength $L]>0 && [dict get [lindex $L end] role] eq "user"} {
                set ::teuk::chatgpt::hist($target) [lrange $L 0 end-1]
            }
        }
        return
    }
    if {[catch {set answer [::teuk::chatgpt::_parse_answer $resp]} derr]} {
        ::teuk::chatgpt::dlog 1 "parse error: $derr"
        puthelp "NOTICE $nick :Réponse API illisible."
        # rollback user
        if {[info exists ::teuk::chatgpt::hist($target)]} {
            set L $::teuk::chatgpt::hist($target)
            if {[llength $L]>0 && [dict get [lindex $L end] role] eq "user"} {
                set ::teuk::chatgpt::hist($target) [lrange $L 0 end-1]
            }
        }
        return
    }

    set answer [regsub -all {[\r\n]+} $answer " "]
    set answer [regsub -all {\s{2,}} $answer " "]

    # Ajoute assistant à l'historique et re-trim
    ::teuk::chatgpt::_hist_add $target assistant $answer

    set chunks [::teuk::chatgpt::_wrap_chunks $answer]
    ::teuk::chatgpt::_send_chunks $target $chunks
}

# 10) Binds publics/privés
proc ::teuk::chatgpt::cmd_chatgpt_pubm {nick uhost hand chan text} {
    if {![channel get $chan chatgpt]} { return }
    if {![regexp -nocase -- {^!chatgpt\s+(.+)$} $text -> args]} { return }
    ::teuk::chatgpt::_handle_prompt $nick $chan $args
}
bind pubm -|- * ::teuk::chatgpt::cmd_chatgpt_pubm

proc ::teuk::chatgpt::cmd_chatgpt_msg {nick uhost hand text} {
    if {$text eq ""} { puthelp "NOTICE $nick :Syntaxe: chatgpt <prompt>" ; return }
    ::teuk::chatgpt::_handle_prompt $nick $nick $text
}
bind msg -|- chatgpt ::teuk::chatgpt::cmd_chatgpt_msg

# 11) Commandes auxiliaires (admin/tests)
# !gptreset           -> efface l'historique du target
# !gptset hist <n>    -> fixe fenêtre messages pour target courant
# !gptset chars <n>   -> fixe budget chars pour target courant
# !gptecho <role> <text> -> injecte dans l'historique (tests): role=user|assistant
proc ::teuk::chatgpt::cmd_aux_pubm {nick uhost hand chan text} {
    if {![channel get $chan chatgpt]} { return }
    if {[regexp -nocase -- {^!gptreset\s*$} $text]} {
        unset -nocomplain ::teuk::chatgpt::hist($chan)
        puthelp "PRIVMSG $chan :[format "\u2705 historique réinitialisé pour %s" $chan]"
        return
    }
    if {[regexp -nocase -- {^!gptset\s+hist\s+(\d+)\s*$} $text -> n]} {
        set ::teuk::chatgpt::hist_overrides($chan) [dict merge [dict create] [expr {[info exists ::teuk::chatgpt::hist_overrides($chan)] ? $::teuk::chatgpt::hist_overrides($chan) : [dict create]}] [dict create max_msgs $n]]
        puthelp "PRIVMSG $chan :[format "\u2705 hist_max_msgs=%d pour %s" $n $chan]"
        return
    }
    if {[regexp -nocase -- {^!gptset\s+chars\s+(\d+)\s*$} $text -> n]} {
        set ::teuk::chatgpt::hist_overrides($chan) [dict merge [dict create] [expr {[info exists ::teuk::chatgpt::hist_overrides($chan)] ? $::teuk::chatgpt::hist_overrides($chan) : [dict create]}] [dict create max_chars $n]]
        puthelp "PRIVMSG $chan :[format "\u2705 hist_max_chars=%d pour %s" $n $chan]"
        return
    }
    if {[regexp -nocase -- {^!gptecho\s+(user|assistant)\s+(.+)$} $text -> role payload]} {
        ::teuk::chatgpt::_hist_add $chan $role $payload
        puthelp "PRIVMSG $chan :[format "\u2139\uFE0F ajouté au contexte (%s): %s" $role [string range $payload 0 80]]"
        return
    }
}
bind pubm -|- * ::teuk::chatgpt::cmd_aux_pubm

# Version PM des commandes auxiliaires
proc ::teuk::chatgpt::cmd_aux_msg {nick uhost hand text} {
    if {[regexp -nocase -- {^gptreset\s*$} $text]} {
        unset -nocomplain ::teuk::chatgpt::hist($nick)
        puthelp "NOTICE $nick :Historique réinitialisé pour $nick"
        return
    }
    if {[regexp -nocase -- {^gptset\s+hist\s+(\d+)\s*$} $text -> n]} {
        set ::teuk::chatgpt::hist_overrides($nick) [dict merge [dict create] [expr {[info exists ::teuk::chatgpt::hist_overrides($nick)] ? $::teuk::chatgpt::hist_overrides($nick) : [dict create]}] [dict create max_msgs $n]]
        puthelp "NOTICE $nick :hist_max_msgs=$n pour $nick"
        return
    }
    if {[regexp -nocase -- {^gptset\s+chars\s+(\d+)\s*$} $text -> n]} {
        set ::teuk::chatgpt::hist_overrides($nick) [dict merge [dict create] [expr {[info exists ::teuk::chatgpt::hist_overrides($nick)] ? $::teuk::chatgpt::hist_overrides($nick) : [dict create]}] [dict create max_chars $n]]
        puthelp "NOTICE $nick :hist_max_chars=$n pour $nick"
        return
    }
    if {[regexp -nocase -- {^gptecho\s+(user|assistant)\s+(.+)$} $text -> role payload]} {
        ::teuk::chatgpt::_hist_add $nick $role $payload
        puthelp "NOTICE $nick :ajout contexte ($role): [string range $payload 0 80]"
        return
    }
}
bind msg -|- gptreset ::teuk::chatgpt::cmd_aux_msg
bind msg -|- gptset   ::teuk::chatgpt::cmd_aux_msg
bind msg -|- gptecho  ::teuk::chatgpt::cmd_aux_msg

# 12) Setters .tcl (partyline)
proc teuk::chatgpt::setkey {k} { set ::teuk::chatgpt::api_key $k }
proc teuk::chatgpt::setsend {m} {
    set ::teuk::chatgpt::send_mode unicode
    return "OK send_mode=unicode"
}
proc teuk::chatgpt::setmode {m} {
    if {!$::teuk::chatgpt::HAVE_TLS} { return "TLS indisponible" }
    set m [string tolower [string trim $m]]
    if {$m ni {wrapper direct}} { return "usage: .tcl eval teuk::chatgpt::setmode wrapper|direct" }
    if {$m eq "wrapper"} {
        set ::teuk::chatgpt::use_wrapper_https 1
        ::teuk::chatgpt::dlog 1 "mode=wrapper (schéma httpsgpt:// actif)"
        return "OK wrapper"
    } else {
        set ::teuk::chatgpt::use_wrapper_https 0
        ::teuk::chatgpt::dlog 1 "mode=direct (schéma https://)"
        return "OK direct"
    }
}

# 13) Diag HTTP/TLS rapide
proc teuk::chatgpt::diag {} {
    if {!$::teuk::chatgpt::HAVE_TLS} { return "diag: TLS indisponible" }
    set host "api.openai.com"
    set port 443
    if {[catch { set s [::teuk::chatgpt::_tls_socket -async 0 $host $port] } e]} {
        return "diag: _tls_socket error $host:$port -> $e"
    }
    fconfigure $s -translation binary -encoding binary -blocking 1 -buffering none
    set req "HEAD /v1/models HTTP/1.1\r\nHost: $host\r\nConnection: close\r\nAccept: */*\r\n\r\n"
    catch { puts -nonewline $s $req; flush $s }
    set line ""
    set deadline [expr {[clock milliseconds]+5000}]
    while {![eof $s]} {
        if {[clock milliseconds] > $deadline} { break }
        if {[gets $s line] >= 0 && $line ne ""} { break }
        after 50
    }
    catch { close $s }
    return "diag: first-line='$line'"
}

catch { putlog [format {chatgpt v%s by Te[u]K a été chargé} $::teuk::chatgpt::script_version] }
