# chatGPT.tcl — Eggdrop + OpenAI (Tcl 8.6+, http+tls, json optionnel)
# Public: !chatgpt <prompt>  (chanset +chatGPT)
# Privé : /msg <BotNick> chatgpt <prompt>

# 0) Version & Debug
namespace eval ::teuk::chatgpt {
    variable script_version "1.2.5"
    variable debug 2  ;# 0=min, 1=normal, 2=très verbeux
}
proc ::teuk::chatgpt::dlog {lvl msg} {
    if {$lvl <= $::teuk::chatgpt::debug} {
        putlog "chatgpt v$::teuk::chatgpt::script_version d$lvl: $msg"
    }
}

# 1) Namespace
namespace eval ::teuk::chatgpt {}

# 2) Packages
if {[catch {package require http} perr]} { error "chatgpt: package http requis: $perr" }
::teuk::chatgpt::dlog 1 "http pkg: [package provide http]"

# JSON (optionnel)
set ::teuk::chatgpt::HAVE_JSON 1
if {[catch {package require json} jerr]} {
    set ::teuk::chatgpt::HAVE_JSON 0
    ::teuk::chatgpt::dlog 1 "json pkg: ABSENT ($jerr) — fallback sans package json"
} else {
    ::teuk::chatgpt::dlog 1 "json pkg: [package provide json]"
}

# TLS
if {[catch {package require tls} terr]} {
    putlog "chatgpt: WARNING: package tls absent (HTTPS requis)"
    set ::teuk::chatgpt::HAVE_TLS 0
    ::teuk::chatgpt::dlog 1 "tls pkg: ABSENT ($terr)"
} else {
    set ::teuk::chatgpt::HAVE_TLS 1
    ::teuk::chatgpt::dlog 1 "tls pkg: [package provide tls]"
    set _tlsinit [catch { ::tls::init -autoservername 1 -tls1.3 1 -tls1.2 1 } _tlsinit_err]
    if {$_tlsinit} {
        ::teuk::chatgpt::dlog 1 "tls::init error: $_tlsinit_err"
    } else {
        ::teuk::chatgpt::dlog 2 "tls::init OK: autoservername=1 tls1.3=1 tls1.2=1"
    }
}

# User-Agent HTTP
catch { ::http::config -useragent "Eggdrop-ChatGPT/1.0 (+tcl)" }
::teuk::chatgpt::dlog 2 "http user-agent configuré"

# 2bis) Options runtime
namespace eval ::teuk::chatgpt {
    variable use_wrapper_https 1
    variable http_decode_utf8 1
    # Mode d’envoi: on force "unicode" (pas de conversions exotiques)
    variable send_mode unicode

    # Découpage & throttling IRC
    variable wrap_bytes   420
    variable max_privmsg  6
    variable trunc_msg    " ...<truncated>"
    variable sleep_ms     600
}
proc ::teuk::chatgpt::_ms {} { return [clock milliseconds] }

# 3) Config API
namespace eval ::teuk::chatgpt {
    variable api_key      "YOUR_API_KEY_HERE"
    variable api_url      "https://api.openai.com/v1/chat/completions"
    variable model        "gpt-4o-mini"
    variable temperature  0.5
    variable max_tokens   400

    catch { setudef flag chatgpt }
}

# 4) HTTPS SNI wrapper
proc ::teuk::chatgpt::_tls_socket {args} {
    set opts {}; set host ""; set port ""
    foreach a $args {
        if {[string match -* $a]} {
            lappend opts $a
        } elseif {$host eq ""} { set host $a
        } elseif {$port eq ""} { set port $a }
    }
    if {$host eq "" || $port eq ""} {
        ::teuk::chatgpt::dlog 1 "_tls_socket args invalides: $args"
        error "_tls_socket: args invalides: $args"
    }
    ::teuk::chatgpt::dlog 2 "_tls_socket host=$host port=$port opts=$opts"
    if {[catch { set sock [::tls::socket {*}$opts -autoservername 1 -servername $host $host $port] } err]} {
        ::teuk::chatgpt::dlog 1 "tls::socket error $host:$port -> $err (opts=$opts)"
        return -code error $err
    }
    return $sock
}
if {$::teuk::chatgpt::HAVE_TLS} {
    set _reg [catch { ::http::register https 443 ::teuk::chatgpt::_tls_socket } _regerr]
    if {$_reg} { ::teuk::chatgpt::dlog 1 "http::register https failed: $_regerr" } \
    else { ::teuk::chatgpt::dlog 2 "http::register https -> wrapper OK" }
}

# 5) JSON helpers
proc ::teuk::chatgpt::_make_payload {prompt} {
    variable model; variable temperature; variable max_tokens
    # >>> consigne modifiée : "no emojis in the answer"
    set sys "You always answer in a helpfull and serious way , precise and never start your answer with « Oh là là » when the answer is in french, always respond using a maximum of 10 lines of text and line-based and no émojis in the answer"
    set prompt [string trim $prompt]

    if {$::teuk::chatgpt::HAVE_JSON} {
        set jmodel  [json::write string $model]
        set jsys    [json::write string $sys]
        set jprompt [json::write string $prompt]
        set msg1 [json::write object role [json::write string system] content $jsys]
        set msg2 [json::write object role [json::write string user]   content $jprompt]
        set messages [json::write array $msg1 $msg2]
        set payload [json::write object model $jmodel temperature $temperature max_tokens $max_tokens messages $messages]
        ::teuk::chatgpt::dlog 2 "payload(json) => $payload"
        return $payload
    } else {
        set p [::teuk::chatgpt::_json_escape [regsub -all {\s+} $prompt " "]]
        set s [::teuk::chatgpt::_json_escape $sys]
        set template {{"model":"%MODEL%","temperature":%TEMP%,"max_tokens":%TOK%,"messages":[{"role":"system","content":"%SYS%"},{"role":"user","content":"%USR%"}]}}
        set payload [string map [list %MODEL% $model %TEMP% $temperature %TOK% $max_tokens %SYS% $s %USR% $p] $template]
        ::teuk::chatgpt::dlog 2 "payload => $payload"
        return $payload
    }
}
proc ::teuk::chatgpt::_json_escape {s} {
    set mapping [list "\\" "\\\\" "\"" "\\\"" "\r" "\\r" "\n" "\\n" "\t" "\\t"]
    return [string map $mapping $s]
}

# 6) Appel HTTP
proc ::teuk::chatgpt::_call_api {payload} {
    variable api_key; variable api_url
    variable http_decode_utf8
    if {$api_key eq ""} { return -code error "cle API manquante" }
    if {!$::teuk::chatgpt::HAVE_TLS} { return -code error "TLS non disponible (installe tcl-tls)" }

    set headers [list Authorization "Bearer $api_key" Content-Type "application/json; charset=utf-8" Accept "application/json"]
    set body    [encoding convertto utf-8 $payload]

    set t0 [::teuk::chatgpt::_ms]
    ::teuk::chatgpt::dlog 2 "http::geturl -> $api_url (POST, timeout=30000, strict=1, binary=1)"
    if {[catch {
        set tok [::http::geturl $api_url -method POST -headers $headers -type "application/json; charset=utf-8" -query $body -timeout 30000 -strict 1 -binary 1]
    } err]} {
        set dt [expr {[::teuk::chatgpt::_ms]-$t0}]
        ::teuk::chatgpt::dlog 1 "http geturl error -> $err (dt=${dt}ms)"
        return -code error "http geturl failed: $err"
    }

    set stat [::http::status $tok]
    set code [::http::ncode  $tok]
    set dt [expr {[::teuk::chatgpt::_ms]-$t0}]
    ::teuk::chatgpt::dlog 2 "http status=$stat ncode=$code dt=${dt}ms"

    set raw [::http::data $tok]
    ::http::cleanup $tok

    if {$http_decode_utf8} {
        if {[catch { set raw [encoding convertfrom utf-8 $raw] } ce]} {
            ::teuk::chatgpt::dlog 1 "convertfrom utf-8 failed: $ce"
        }
    }
    set data $raw
    ::teuk::chatgpt::dlog 2 "raw <= [string range $data 0 300]..."

    if {$stat ne "ok"} { return -code error "http $stat (code $code)" }
    if {$code < 200 || $code >= 300} { return -code error "http $code: $data" }
    return $data
}

# 7) Parsing JSON
proc ::teuk::chatgpt::_parse_answer {response} {
    if {$::teuk::chatgpt::HAVE_JSON} {
        set d [json::json2dict $response]
        if {![dict exists $d choices]} { return -code error "invalid JSON: no choices" }
        set choices [dict get $d choices]
        if {[llength $choices] < 1} { return -code error "invalid JSON: empty choices" }
        set first [lindex $choices 0]
        if {![dict exists $first message]} { return -code error "invalid JSON: no message" }
        set msg [dict get $first message]
        if {![dict exists $msg content]} { return -code error "invalid JSON: no content" }
        set content [dict get $msg content]
        return [string trim $content]
    } else {
        if {[regexp {\"content\"\s*:\s*\"((?:\\.|[^\"\\])*)\"} $response -> m]} {
            set mapping [list "\\n" "\n" "\\r" "\r" "\\t" "\t" "\\\\" "\\" "\\\"" "\""]
            set m [string map $mapping $m]
            return [string trim $m]
        }
        return -code error "invalid JSON response"
    }
}

# 8) Wrap helper
proc ::teuk::chatgpt::_wrap_chunks {txt} {
    variable wrap_bytes
    set out {}
    set txt [string trim $txt]
    while {[string length $txt] > 0} {
        if {[string length $txt] <= $wrap_bytes} { lappend out $txt ; break }
        set slice [string range $txt 0 [expr {$wrap_bytes-1}]]
        set brk [string last " " $slice]
        if {$brk < 0} { set brk [expr {$wrap_bytes-1}] }
        lappend out [string trimright [string range $txt 0 $brk]]
        set txt [string trimleft [string range $txt [expr {$brk+1}] end]]
    }
    return $out
}

# ====== DEBUG helpers ======
proc ::teuk::chatgpt::_hex_utf8 {s {n 64}} {
    set b [encoding convertto utf-8 $s]
    return [binary encode hex [string range $b 0 [expr {$n-1}]]]
}
proc ::teuk::chatgpt::_dbg_show {label s} {
    set clen [string length $s]
    set b [encoding convertto utf-8 $s]
    set blen [string length $b]
    set hex [binary encode hex [string range $b 0 63]]
    ::teuk::chatgpt::dlog 2 "$label CHAR=$clen UTF8=$blen HEX\[0..63]=$hex"
    # Simulation double-encodage pour diagnostic
    set s_l1 [encoding convertfrom iso8859-1 $b]
    set b2   [encoding convertto utf-8 $s_l1]
    set hex2 [binary encode hex [string range $b2 0 63]]
    ::teuk::chatgpt::dlog 2 "$label SIMU-DOUBLE HEX\[0..63]=$hex2"
}

# --- ENVOI IRC (UTF-8 pur) ---------------------------------------------------
proc ::teuk::chatgpt::_send_chunks {target chunks} {
    variable max_privmsg; variable trunc_msg; variable sleep_ms
    variable send_mode

    # On force de toute façon l'envoi "unicode"
    set send_mode unicode

    set count [llength $chunks]
    set last [expr {$count-1}]
    if {$count > $max_privmsg} {
        set last [expr {$max_privmsg-1}]
        set lastChunk [lindex $chunks $last]
        set allow [expr {[string length $lastChunk] - [string length $trunc_msg]}]
        if {$allow < 1} { set allow 1 }
        set chopped [string range $lastChunk 0 $allow]
        set space [string last " " $chopped]
        if {$space > 0} { set chopped [string range $chopped 0 $space] }
        set chopped [string trimright $chopped]
        set chunks [lreplace $chunks $last $last "${chopped}${trunc_msg}"]
    }

    for {set i 0} {$i <= $last} {incr i} {
        set line [lindex $chunks $i]
        if {$line eq ""} { continue }
        set msg "PRIVMSG $target :$line"

        # Debug fort : on loggue les octets UTF-8 qu'on VA envoyer
        set want [encoding convertto utf-8 $msg]
        set wanthex [binary encode hex [string range $want 0 63]]
        ::teuk::chatgpt::dlog 2 "SEND(unicode) WILL-SEND HEX\[0..63]=$wanthex"
        ::teuk::chatgpt::_dbg_show "SEND(pre)" $msg

        # Envoi "propre" (laisser Eggdrop/Tcl encoder)
        puthelp $msg
        after $sleep_ms
    }
    ::teuk::chatgpt::dlog 1 "sent [expr {$last+1}] PRIVMSG to $target (mode=unicode)"
}

# 9) Pipeline
proc ::teuk::chatgpt::_handle_prompt {nick target prompt} {
    ::teuk::chatgpt::dlog 1 "prompt from $nick: $prompt"
    ::teuk::chatgpt::_dbg_show "RECV(prompt)" $prompt

    set prompt [regsub -all {\s+} [string trim $prompt] " "]
    if {$prompt eq ""} { puthelp "NOTICE $nick :Syntaxe: !chatgpt <prompt>" ; return }

    if {[catch {set payload [::teuk::chatgpt::_make_payload $prompt]} perr]} {
        ::teuk::chatgpt::dlog 1 "payload error: $perr"
        puthelp "NOTICE $nick :Erreur interne (payload)"
        return
    }
    if {[catch {set resp [::teuk::chatgpt::_call_api $payload]} herr]} {
        ::teuk::chatgpt::dlog 1 $herr
        puthelp "NOTICE $nick :API indisponible."
        return
    }
    if {[catch {set answer [::teuk::chatgpt::_parse_answer $resp]} derr]} {
        ::teuk::chatgpt::dlog 1 "parse error: $derr"
        puthelp "NOTICE $nick :Réponse API illisible."
        return
    }

    set answer [regsub -all {[\r\n]+} $answer " "]
    set answer [regsub -all {\s{2,}} $answer " "]
    ::teuk::chatgpt::_dbg_show "ANS(normalized)" $answer

    set chunks [::teuk::chatgpt::_wrap_chunks $answer]
    ::teuk::chatgpt::_send_chunks $target $chunks
}

# 10) Binds
proc ::teuk::chatgpt::cmd_chatgpt_pubm {nick uhost hand chan text} {
    if {![channel get $chan chatgpt]} { return }
    ::teuk::chatgpt::dlog 1 "pubm recv on $chan from $nick: $text"
    if {![regexp -nocase -- {^!chatgpt\s+(.+)$} $text -> args]} { return }
    ::teuk::chatgpt::_handle_prompt $nick $chan $args
}
bind pubm -|- * ::teuk::chatgpt::cmd_chatgpt_pubm

proc ::teuk::chatgpt::cmd_chatgpt_msg {nick uhost hand text} {
    if {$text eq ""} { puthelp "NOTICE $nick :Syntaxe: chatgpt <prompt>" ; return }
    ::teuk::chatgpt::_handle_prompt $nick $nick $text
}
bind msg -|- chatgpt ::teuk::chatgpt::cmd_chatgpt_msg

# 11) Setters .tcl (partyline)
proc teuk::chatgpt::setkey {k} { set ::teuk::chatgpt::api_key $k }
proc teuk::chatgpt::setsend {m} {
    # conservé pour compat ; on force "unicode" de toute façon
    set ::teuk::chatgpt::send_mode unicode
    return "OK send_mode=unicode"
}
proc teuk::chatgpt::setmode {m} {
    if {!$::teuk::chatgpt::HAVE_TLS} { return "TLS indisponible" }
    set m [string tolower [string trim $m]]
    if {$m ni {wrapper direct}} { return "usage: .tcl eval teuk::chatgpt::setmode wrapper|direct" }
    if {$m eq "wrapper"} {
        set ::teuk::chatgpt::use_wrapper_https 1
        set _reg [catch { ::http::register https 443 ::teuk::chatgpt::_tls_socket } _regerr]
        if {$_reg} { return "register wrapper: $_regerr" }
        ::teuk::chatgpt::dlog 1 "mode=wrapper (_tls_socket)"
        return "OK wrapper"
    } else {
        set ::teuk::chatgpt::use_wrapper_https 0
        set _reg [catch { ::http::register https 443 ::tls::socket } _regerr]
        if {$_reg} { return "register direct: $_regerr" }
        ::teuk::chatgpt::dlog 1 "mode=direct (::tls::socket)"
        return "OK direct"
    }
}

# 12) Diag HTTP/TLS rapide
proc teuk::chatgpt::diag {} {
    if {!$::teuk::chatgpt::HAVE_TLS} { return "diag: TLS indisponible" }
    set host "api.openai.com"
    set port 443
    if {[catch { set s [::tls::socket -autoservername 1 -servername $host $host $port] } e]} {
        return "diag: tls::socket error $host:$port -> $e"
    }
    fconfigure $s -translation binary -encoding binary -blocking 1 -buffering none
    set req "HEAD /v1/models HTTP/1.1\r\nHost: $host\r\nConnection: close\r\nAccept: */*\r\n\r\n"
    catch { puts -nonewline $s $req; flush $s }
    set line ""
    set deadline [expr {[clock milliseconds]+5000}]
    while {![eof $s]} {
        if {[clock milliseconds] > $deadline} { break }
        if {[gets $s line] >= 0 && $line ne ""} { break }
        after 50
    }
    catch { close $s }
    return "diag: first-line='$line'"
}
