########################################################################################
# DansTonChat / BashFR pour Eggdrop
# v1.14.7 (2025-12-15)   Auteur : TeuK
#
# Commandes :
#   !dtc / !bashfr               -> quote aléatoire
#   !dtc <num> / !bashfr <num>   -> quote par numéro
#   !dtc <texte>                 -> recherche (DuckDuckGo -> DTC /?s=)
#
# Pré-requis : Eggdrop >= 1.8.4 (build 1080404), Tcl >= 8.6
# Chanset    : +DansTonChat
#
# CHANGELOG (résumé)
#   1.14.7 : unregister du protocole httpsdtc au unload, fix check version Eggdrop,
#            limite à 10 lignes + ajout "... / Consultez la quote ici".
#   1.14.6 : Intégration des remarques de MenzAgitat (version Eggdrop, conf dans ns).
#   1.14.5 : Nettoyage namespace + logs.
#   1.14.4 : Fix du /random (vrai aléatoire).
#   1.14.0 : Simplification générale, plus de REST WP, recherche DDG+fallback.
########################################################################################

if {[info commands ::dtc::unload] eq "::dtc::unload"} { ::dtc::unload }

# Check Eggdrop build (>= 1080404)
set ::dtc__build [lindex [split $::version] 1]
if {![string is integer -strict $::dtc__build] || $::dtc__build < 1080404} {
    putloglev o * "\00304\[DTC - erreur\]\003 La version de votre Eggdrop est \00304${::version}\003; DTC requiert Eggdrop 1.8.4+ (build 1080404). Chargement annulé."
    unset -nocomplain ::dtc__build
    return
}
unset -nocomplain ::dtc__build

if {[catch { package require Tcl 8.6 }]} {
    putloglev o * "\00304\[DTC - erreur\]\003 DTC nécessite Tcl 8.6+ ; votre version est \00304${::tcl_version}\003. Chargement annulé."
    return
}

# Namespace + procédure d'unload + config
namespace eval ::dtc {
    proc unload {args} {
        set ns [namespace current]
        putlog "DTC: nettoyage du namespace $ns..."

        # unregister du protocole wrapper si on l'a enregistré nous-mêmes
        if {[info exists ::dtc::proto_registered] && $::dtc::proto_registered} {
            catch { ::http::unregister httpsdtc }
            putlog "DTC: protocole httpsdtc unregister (ok/catch)."
        }

        foreach b [binds *] {
            if {[llength $b] < 5} { continue }
            set cmd [lindex $b 4]
            if {[string match "${ns}::*" $cmd]} {
                unbind [lindex $b 0] [lindex $b 1] [lindex $b 2] $cmd
            }
        }
        namespace delete $ns
    }

    variable version "1.14.7"

    # Paramètres de configuration
    variable have_tls           0
    variable use_wrapper_https  1      ;# réécrit https:// -> httpsdtc://
    variable timeout            15000
    variable max_redirects      5
    variable max_lines          10     ;# limite d'affichage (10 lignes max)
    variable debug              0      ;# 0=muet, 1=info, 2=verbeux
    variable line_cooldown_ms   600
    variable http_cooldown_ms   400
    variable id_fg              1      ;# 1 = noir
    variable id_bg              15     ;# 15 = gris clair
    variable quote_fg           0      ;# 0 = blanc
    variable quote_bg           14     ;# 14 = gris sombre
    variable max_search_results 5

    # État runtime
    variable last_send_ts; array set last_send_ts {}
    variable last_http_ts 0

    # État wrapper protocole
    variable proto_registered 0
}

# Seed RNG pour vraiment varier les quotes aléatoires
expr {srand([clock clicks])}

# ---------------------------------------------------------------------------
# Dépendances
# ---------------------------------------------------------------------------
if {[catch {package require http} err]} {
    putlog "DTC: désactivé — package Tcl 'http' manquant ($err)."
    return
}
if {![catch {package require tls} terr]} {
    set ::dtc::have_tls 1
} else {
    putlog "DTC: TLS indisponible, utilisation de la pile réseau système ($terr)."
}
catch {
    ::http::config -useragent "Mozilla/5.0 (X11; Linux x86_64) Chrome/124.0"
}

# ---------------------------------------------------------------------------
# Logs et utilitaires
# ---------------------------------------------------------------------------
proc ::dtc::log {niveau tag msg} {
    if {$::dtc::debug >= $niveau} { putlog "DTC($tag): $msg" }
}
proc ::dtc::log_info  {msg} { ::dtc::log 1 INFO   $msg }
proc ::dtc::log_err   {msg} { ::dtc::log 0 ERREUR $msg }
proc ::dtc::log_http  {msg} { ::dtc::log 2 HTTP   $msg }
proc ::dtc::log_parse {msg} { ::dtc::log 2 PARSE  $msg }

proc ::dtc::rate_send {chan line} {
    set now [clock milliseconds]
    if {![::info exists ::dtc::last_send_ts($chan)]} { set ::dtc::last_send_ts($chan) 0 }
    set delta [expr {$now - $::dtc::last_send_ts($chan)}]
    if {$delta < $::dtc::line_cooldown_ms} { after [expr {$::dtc::line_cooldown_ms - $delta}] }
    puthelp "PRIVMSG $chan :$line"
    set ::dtc::last_send_ts($chan) [clock milliseconds]
}

# ---------------------------------------------------------------------------
# Wrapper TLS minimal + réécriture URL
# ---------------------------------------------------------------------------
proc ::dtc::_tls_socket {args} {
    set async 0; set host ""; set port ""; set myaddr ""; set myport ""; set keepalive ""
    set i 0; set n [llength $args]
    while {$i < $n} {
        set a [lindex $args $i]
        if {[string match -* $a]} {
            switch -- $a {
                -async     { set async 1 }
                -myaddr    { incr i; set myaddr  [lindex $args $i] }
                -myport    { incr i; set myport  [lindex $args $i] }
                -keepalive { set keepalive 1 }
                default {}
            }
        } elseif {$host eq ""} { set host $a } elseif {$port eq ""} { set port $a }
        incr i
    }
    if {$host eq "" || $port eq ""} { error "_tls_socket: arguments invalides: $args" }
    set opts {}
    if {$myaddr ne ""}    { lappend opts -myaddr $myaddr }
    if {$myport ne ""}    { lappend opts -myport $myport }
    if {$keepalive ne ""} { lappend opts -keepalive $keepalive }
    set s [::socket {*}$opts $host $port]
    catch { fconfigure $s -translation binary -encoding binary -buffering none }
    if {$async} { catch { fconfigure $s -blocking 0 } }
    set importe 0
    foreach essai {
        {-autoservername 1 -servername __HOST__}
        {-servername __HOST__}
        {}
    } {
        set cmd [list ::tls::import $s]
        if {[llength $essai]} {
            foreach {k v} $essai {
                if {$v eq "__HOST__"} { set v $host }
                lappend cmd $k $v
            }
        }
        if {![catch { {*}$cmd }]} { set importe 1; break }
    }
    if {!$importe} { catch {close $s}; error "tls::import a échoué pour $host:$port" }
    return $s
}

# Register du protocole wrapper (global). Unregister dans unload (si on est l'auteur du register).
if {$::dtc::have_tls} {
    if {![catch { ::http::register httpsdtc 443 ::dtc::_tls_socket } regerr]} {
        set ::dtc::proto_registered 1
        ::dtc::log_info "protocole httpsdtc enregistré via http::register"
    } else {
        # Déjà enregistré (rehash/2 scripts) ou autre souci : on log, mais on ne casse pas tout.
        ::dtc::log_info "protocole httpsdtc non enregistré (déjà là ou erreur): $regerr"
        set ::dtc::proto_registered 0
    }
}

proc ::dtc::_rewrite_url {url} {
    if {$::dtc::have_tls && $::dtc::use_wrapper_https && [string match -nocase "https://*" $url]} {
        return [string map {"https://" "httpsdtc://"} $url]
    }
    return $url
}

# ---------------------------------------------------------------------------
# HTTP GET simple
# ---------------------------------------------------------------------------
proc ::dtc::fetch {url {depth 0}} {
    if {$depth > $::dtc::max_redirects} {
        ::dtc::log_err "trop de redirections pour $url"
        return [list 0 "" {}]
    }
    set now [clock milliseconds]
    set delta [expr {$now - $::dtc::last_http_ts}]
    if {$delta < $::dtc::http_cooldown_ms} { after [expr {$::dtc::http_cooldown_ms - $delta}] }
    set eff [::dtc::_rewrite_url $url]
    ::dtc::log_http "GET $eff (profondeur=$depth)"
    if {[catch {::http::geturl $eff -timeout $::dtc::timeout} token]} {
        ::dtc::log_err "erreur HTTP sur $eff"
        return [list 0 "" {}]
    }
    set code [::http::ncode $token]
    set body [::http::data $token]
    set meta [::http::meta $token]
    ::http::cleanup $token
    set ::dtc::last_http_ts [clock milliseconds]
    ::dtc::log_http "HTTP $code, octets=[string length $body]"
    if {$code >= 300 && $code < 400} {
        set loc ""
        for {set i 0} {$i < [llength $meta]} {incr i 2} {
            if {[string tolower [lindex $meta $i]] eq "location"} {
                set loc [lindex $meta [expr {$i+1}]]
                break
            }
        }
        if {$loc ne ""} {
            if {![regexp {^https?://|^httpsdtc://} $loc]} {
                if {[regexp {^(https?://[^/]+)} $url -> origin]} {
                    if {[string index $loc 0] ne "/"} { set loc "/$loc" }
                    set loc "$origin$loc"
                }
            }
            ::dtc::log_http "redirection -> $loc"
            return [::dtc::fetch $loc [expr {$depth+1}]]
        }
    }
    return [list $code $body $meta]
}

# ---------------------------------------------------------------------------
# Helpers HTML
# ---------------------------------------------------------------------------
proc ::dtc::html_named_unescape {s} {
    set map [list "&amp;" "&" "&lt;" "<" "&gt;" ">" "&quot;" "\"" "&apos;" "'" "&nbsp;" " " \
                  "&laquo;" "«" "&raquo;" "»" "&hellip;" "…" "&ndash;" "–" "&mdash;" "—" \
                  "&lsquo;" "‘" "&rsquo;" "’" "&ldquo;" "“" "&rdquo;" "”" "&oelig;" "œ" "&OElig;" "Œ" "&euro;" "€"]
    return [string map $map $s]
}
proc ::dtc::decode_numeric_entities {s} {
    while {[regexp -indices {&#([0-9]+);} $s m n]} {
        foreach {m0 m1} $m {}
        foreach {n0 n1} $n {}
        set num [string range $s $n0 $n1]
        if {[catch {set ch [format %c $num]}]} { set ch "?" }
        set s [string replace $s $m0 $m1 $ch]
    }
    while {[regexp -indices {&#[xX]([0-9A-Fa-f]+);} $s m n]} {
        foreach {m0 m1} $m {}
        foreach {n0 n1} $n {}
        set hex [string range $s $n0 $n1]
        if {[catch {set ch [format %c 0x$hex]}]} { set ch "?" }
        set s [string replace $s $m0 $m1 $ch]
    }
    return $s
}
proc ::dtc::html_unescape {s} {
    set s [::dtc::html_named_unescape $s]
    set s [::dtc::decode_numeric_entities $s]
    return $s
}

# ---------------------------------------------------------------------------
# Extraction de la quote principale
# ---------------------------------------------------------------------------
proc ::dtc::extract_first_quote {html} {
    if {$html eq ""} { return "" }
    set html [string map {"\r\n" "\n" "\r" "\n"} $html]
    if {![regexp -nocase -indices {<div[^>]*class="[^"]*entry-content[^"]*"[^>]*>} $html idx]} { return "" }
    foreach {openS openE} $idx {}
    set gt [string first ">" $html $openS]
    if {$gt < 0} { return "" }
    set start [expr {$gt + 1}]
    set fin -1
    if {[regexp -indices -nocase -start $start {</div>\s*</div>} $html closeIdx]} {
        foreach {cs ce} $closeIdx {}
        set fin [expr {$cs - 1}]
    } else {
        set pos $start
        set depth 1
        while {$depth > 0} {
            set o [string first "<div"   $html $pos]
            set c [string first "</div>" $html $pos]
            if {$c < 0 && $o < 0} { break }
            if {$o >= 0 && ($o < $c || $c < 0)} {
                set pos [expr {$o+4}]
                incr depth
            } else {
                set fin [expr {$c-1}]
                set pos [expr {$c+6}]
                incr depth -1
            }
            if {$pos >= [string length $html]} { break }
        }
    }
    if {$fin <= $start} { return "" }
    set block [string range $html $start $fin]
    regsub -all -nocase {<(br|/p|/li)\s*[^>]*>} $block "\n" block
    regsub -all -nocase {<p[^>]*>}             $block ""   block
    regsub -all -nocase {<li[^>]*>}            $block "• " block
    regsub -all {<[^>]*>}                      $block ""   block
    set block [::dtc::html_unescape [string trim $block]]
    set lignes [split $block "\n"]
    set garde {}
    foreach L $lignes {
        set L [string trim $L]
        if {$L eq ""} { continue }
        if {[string trim $L] eq "•"} { continue }
        lappend garde $L
    }
    if {[llength $garde] == 0} { return "" }
    return [join $garde "\n"]
}

# ---------------------------------------------------------------------------
# Recherche : DuckDuckGo + fallback DTC
# ---------------------------------------------------------------------------
proc ::dtc::urldecode {s} {
    set out ""
    set i 0
    set n [string length $s]
    while {$i < $n} {
        set c [string index $s $i]
        if {$c eq "%"} {
            if {$i+2 < $n} {
                set hex [string range $s [expr {$i+1}] [expr {$i+2}]]
                if {[regexp {^[0-9A-Fa-f]{2}$} $hex]} {
                    scan $hex %x code
                    append out [format %c $code]
                    incr i 3
                    continue
                }
            }
        } elseif {$c eq "+"} {
            append out " "
            incr i
            continue
        }
        append out $c
        incr i
    }
    return $out
}
proc ::dtc::url_search_ddg_narrow {q} {
    set query [format "site:danstonchat.com/quote %s" $q]
    set qs [::http::formatQuery q $query kl fr-fr kp -2 ia web]
    return "https://duckduckgo.com/html/?$qs"
}
proc ::dtc::url_search_ddg_lite {q} {
    set query [format "site:danstonchat.com %s" $q]
    set qs [::http::formatQuery q $query]
    return "https://lite.duckduckgo.com/lite/?$qs"
}
proc ::dtc::parse_ddg_ids {html} {
    set ids {}
    array set vu {}
    set start 0
    while {[regexp -indices -nocase -start $start {(?:https?://(?:www\.)?danstonchat\.com)?/quote/([0-9]+)\.html} $html m]} {
        foreach {s e} $m {}
        set frag [string range $html $s $e]
        if {[regexp {([0-9]+)} $frag -> id]} {
            if {![info exists vu($id)]} {
                set vu($id) 1
                lappend ids $id
            }
        }
        set start [expr {$e + 1}]
    }
    set start 0
    while {[regexp -indices -nocase -start $start {uddg=([^"'&]+)} $html m uIdx]} {
        foreach {s e} $m {}
        foreach {us ue} $uIdx {}
        set dec [::dtc::urldecode [string range $html $us $ue]]
        if {[regexp -nocase {/quote/([0-9]+)\.html} $dec -> id]} {
            if {![info exists vu($id)]} {
                set vu($id) 1
                lappend ids $id
            }
        }
        set start [expr {$e + 1}]
    }
    return $ids
}
proc ::dtc::search_ids {q} {
    set ids {}
    set url1 [::dtc::url_search_ddg_narrow $q]
    ::dtc::log_info "RECHERCHE DDG narrow : '$q'"
    lassign [::dtc::fetch $url1] code1 html1 _
    if {$code1 == 200 && $html1 ne ""} { set ids [::dtc::parse_ddg_ids $html1] }
    if {[llength $ids] == 0} {
        set url2 [::dtc::url_search_ddg_lite $q]
        ::dtc::log_info "RECHERCHE DDG lite : '$q'"
        lassign [::dtc::fetch $url2] code2 html2 _
        if {$code2 == 200 && $html2 ne ""} { set ids [::dtc::parse_ddg_ids $html2] }
    }
    if {[llength $ids] == 0} {
        set qs [::http::formatQuery s $q]
        set url3 "https://danstonchat.com/?$qs"
        ::dtc::log_info "RECHERCHE DTC fallback : '$q'"
        lassign [::dtc::fetch $url3] code3 html3 _
        if {$code3 == 200 && $html3 ne ""} {
            set start 0
            while {[regexp -indices -nocase -start $start {href=['"][^"']*/quote/([0-9]+)\.html['"]} $html3 m]} {
                foreach {s e} $m {}
                set frag [string range $html3 $s $e]
                if {[regexp {([0-9]+)} $frag -> id]} {
                    if {[lsearch -exact $ids $id] < 0} { lappend ids $id }
                }
                set start [expr {$e + 1}]
            }
        }
    }
    return $ids
}

# ---------------------------------------------------------------------------
# URLs + fetchers
# ---------------------------------------------------------------------------
proc ::dtc::url_random {}   { return "https://danstonchat.com/random" }
proc ::dtc::url_for_id {id} { return "https://danstonchat.com/quote/${id}.html" }

proc ::dtc::fetch_by_id {id} {
    ::dtc::log_info "QUOTE ID $id"
    lassign [::dtc::fetch [::dtc::url_for_id $id]] code html _
    if {$code != 200 || $html eq ""} { return [list "" ""] }
    set q [::dtc::extract_first_quote $html]
    if {$q eq ""} { return [list "" ""] }
    return [list $id [::dtc::strip_trailing_numeric_debris $q]]
}

# Random : une seule requête sur /random, choix aléatoire d'un bloc entry-content
proc ::dtc::fetch_random {} {
    ::dtc::log_info "QUOTE aléatoire"
    lassign [::dtc::fetch [::dtc::url_random]] code html _
    if {$code != 200 || $html eq ""} { return [list "" ""] }

    set html [string map {"\r\n" "\n" "\r" "\n"} $html]
    set blocks {}
    set start 0
    while {[regexp -indices -nocase -start $start {<div[^>]*class="[^"]*entry-content[^"]*"[^>]*>} $html m]} {
        foreach {s e} $m {}
        set gt [string first ">" $html $s]
        if {$gt < 0} { break }
        set cs [expr {$gt+1}]
        set ce -1
        if {[regexp -indices -nocase -start $cs {</div>\s*</div>} $html m2]} {
            foreach {cs2 ce2} $m2 {} ; set ce [expr {$cs2-1}]
        } else {
            set pos $cs
            set depth 1
            while {$depth > 0} {
                set o [string first "<div"   $html $pos]
                set c [string first "</div>" $html $pos]
                if {$c < 0 && $o < 0} { break }
                if {$o >= 0 && ($o < $c || $c < 0)} {
                    set pos [expr {$o+4}]
                    incr depth
                } else {
                    set ce [expr {$c-1}]
                    set pos [expr {$c+6}]
                    incr depth -1
                }
                if {$pos >= [string length $html]} { break }
            }
        }
        if {$ce > $cs} { lappend blocks [list $cs $ce] }
        set start [expr {$e+1}]
    }

    # Si aucun bloc entry-content trouvé, fallback sur IDs + fetch_by_id
    if {[llength $blocks] == 0} {
        set matches [regexp -inline -all -nocase {/quote/([0-9]+)\.html} $html]
        set ids {}
        foreach m $matches {
            if {[regexp {([0-9]+)} $m -> id]} {
                if {[lsearch -exact $ids $id] < 0} { lappend ids $id }
            }
        }
        if {[llength $ids] == 0} { return [list "" ""] }
        set pick [lindex $ids [expr {int(rand()*[llength $ids])}]]
        return [::dtc::fetch_by_id $pick]
    }

    # Choix aléatoire d'un bloc, puis tentative de retrouver l'ID autour
    set blk [lindex $blocks [expr {int(rand()*[llength $blocks])}]]
    foreach {cs ce} $blk {}
    set block [string range $html $cs $ce]

    set rid ""
    set win_start [expr {$cs-1200}]; if {$win_start < 0} { set win_start 0 }
    set win_end   [expr {$ce+1200}]; if {$win_end >= [string length $html]} { set win_end [expr {[string length $html]-1}] }
    set win [string range $html $win_start $win_end]
    if {[regexp -nocase {/quote/([0-9]+)\.html} $win -> qid]} {
        set rid $qid
    } elseif {[regexp -nocase {data-id\s*=\s*"([0-9]+)"} $win -> qid2]} {
        set rid $qid2
    }
    if {$rid eq ""} { set rid "??" }

    set text [::dtc::extract_first_quote "<div class=\"entry-content\">$block</div>"]
    if {$text eq ""} { return [list "" ""] }
    return [list $rid $text]
}

# ---------------------------------------------------------------------------
# Helpers texte + affichage (style !vdm)
# ---------------------------------------------------------------------------
proc ::dtc::strip_trailing_numeric_debris {text} {
    set lignes [split $text "\n"]
    while {[llength $lignes] > 0 && [string trim [lindex $lignes end]] eq ""} {
        set lignes [lreplace $lignes end end]
    }
    if {[llength $lignes] == 0} { return "" }
    set last [string trim [lindex $lignes end]]
    if {[regexp {^[0-9]+$} $last]} {
        set lignes [lreplace $lignes end end]
        return [join $lignes "\n"]
    }
    return $text
}

# Tronquage à max_lines :
# - si > max_lines : on garde (max_lines-2) lignes, puis "..." puis "Consultez la quote ici : <url>"
proc ::dtc::truncate_lines_for_irc {text rid} {
    set raw [split $text "\n"]
    set lines {}
    foreach L $raw {
        set t [string trim $L]
        if {$t eq ""} { continue }
        lappend lines $t
    }

    if {$::dtc::max_lines <= 0 || [llength $lines] <= $::dtc::max_lines} {
        return $lines
    }

    set url ""
    if {[regexp {^[0-9]+$} $rid]} {
        set url "Consultez la quote ici : https://danstonchat.com/quote/${rid}.html"
    }

    set keep [expr {$::dtc::max_lines - 1}]   ;# au minimum on ajoute "..."
    if {$url ne ""} { set keep [expr {$::dtc::max_lines - 2}] }

    if {$keep < 1} { set keep 1 }
    set head [lrange $lines 0 [expr {$keep-1}]]

    lappend head "..."
    if {$url ne ""} { lappend head $url }

    return $head
}

# Première ligne : [ID] gris sur blanc, texte blanc sur gris
proc ::dtc::format_first_line {rid first_line} {
    return "\003$::dtc::id_fg,$::dtc::id_bg\002\[$rid]\002\003$::dtc::quote_fg,$::dtc::quote_bg $first_line\017"
}

# Lignes suivantes : blanc sur gris
proc ::dtc::format_rest_lines {lines} {
    set out {}
    foreach L $lines {
        set L [string trim $L]
        if {$L eq ""} { continue }
        lappend out "\003$::dtc::quote_fg,$::dtc::quote_bg$L\017"
    }
    return $out
}

proc ::dtc::send_quote {chan rid text} {
    set lines [::dtc::truncate_lines_for_irc $text $rid]

    if {[llength $lines] == 0} {
        ::dtc::rate_send $chan "\003$::dtc::id_fg,$::dtc::id_bg\002\[$rid]\002\017"
        return
    }

    set first [lindex $lines 0]
    set rest  [lrange $lines 1 end]

    ::dtc::rate_send $chan [::dtc::format_first_line $rid $first]
    foreach L [::dtc::format_rest_lines $rest] { ::dtc::rate_send $chan $L }
}

# ---------------------------------------------------------------------------
# Chanset + commandes publiques
# ---------------------------------------------------------------------------
setudef flag DansTonChat
bind pub - "!dtc"    ::dtc::cmd
bind pub - "!bashfr" ::dtc::cmd

proc ::dtc::cmd {nick uhost hand chan text} {
    ::dtc::log_info "PUB sur $chan par $nick texte='[string range $text 0 80]'"
    if {![channel get $chan DansTonChat]} {
        puthelp "PRIVMSG $chan :Activez +DansTonChat pour !dtc / !bashfr (.chanset $chan +DansTonChat)."
        return
    }
    set arg [string trim $text]

    # Sans argument -> aléatoire
    if {$arg eq ""} {
        lassign [::dtc::fetch_random] rid q
        if {$q eq ""} {
            ::dtc::rate_send $chan "Désolé, aucune quote aléatoire disponible pour le moment."
            return
        }
        if {$rid eq ""} { set rid "??" }
        ::dtc::send_quote $chan $rid $q
        return
    }

    # Numéro explicitement demandé
    if {[regexp {^\s*\(?\s*([0-9]+)\s*\)?\s*$} $arg -> num]} {
        lassign [::dtc::fetch_by_id $num] rid q
        if {$q eq ""} {
            ::dtc::rate_send $chan "Désolé, je n'ai pas trouvé la quote #$num."
            return
        }
        if {$rid eq ""} { set rid $num }
        ::dtc::send_quote $chan $rid $q
        return
    }

    # Recherche textuelle
    set ids [::dtc::search_ids $arg]
    if {[llength $ids] == 0} {
        ::dtc::rate_send $chan "Aucun résultat pour « $arg »."
        return
    }

    # En-tête avec quelques IDs : texte gris sur fond blanc
    set show_ids [lrange $ids 0 [expr {$::dtc::max_search_results-1}]]
    set header [join $show_ids "|"]
    ::dtc::rate_send $chan "\003$::dtc::id_fg,$::dtc::id_bg\[DansTonChat : $header]\017"

    set rid [lindex $ids 0]
    lassign [::dtc::fetch_by_id $rid] _ q
    if {$q ne ""} {
        ::dtc::send_quote $chan $rid $q
    } else {
        ::dtc::rate_send $chan "Désolé, impossible d'afficher la quote #$rid."
    }
}

# ---------------------------------------------------------------------------
# Fin de chargement
# ---------------------------------------------------------------------------
putlog "DTC: script DansTonChat v$::dtc::version chargé (auteur : TeuK). Commandes !dtc / !bashfr, chanset +DansTonChat requis."
bind evnt - prerehash ::dtc::unload
