# danstonchat.tcl — DansTonChat / BashFR pour Eggdrop
# Cmds :
#   !dtc / !bashfr               -> quote aléatoire (/random)
#   !dtc <num> / !bashfr <num>   -> quote par numéro (/quote/<num>.html)
#   !dtc <terme>                 -> recherche (DDG → WP REST → fallback DTC)
# Chanset requis : +DansTonChat
#
# v1.12.4 — Fix définitif "unbalanced open paren" (if {} en Tcl dans les logs), ID en blanc sur fond gris, TLS wrapper isolé, compat Debian 9/12, sockets fermées, log propre.

# =========================
#        Version
# =========================
set ::dtc(version) "1.12.4"

# =========================
#  Namespace de base (avant toute proc)
# =========================
namespace eval ::dtc {}

# =========================
#     Dépendances TCL
# =========================
if {[catch {package require http} err]} {
    putlog "DTC: désactivé — package 'http' manquant ($err)."
    return
}

set ::dtc(have_tls) 0
set ::dtc(use_wrapper_https) 1  ;# si 1 => réécrit https:// en httpsdtc:// pour nos fetch
if {![catch {package require tls} terr]} {
    set ::dtc(have_tls) 1
} else {
    putlog "DTC: TLS indisponible — on tentera via la pile système ($terr)."
}

# User-Agent
catch {::http::config -useragent "Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/124.0 Safari/537.36"}

# =========================
#    Wrapper TLS isolé
# =========================
proc ::dtc::_tls_socket {args} {
    # Compat Tcl/tls Debian 9/12 — accepte -async, -myaddr, -myport, -keepalive, puis host port.
    set async 0
    set myaddr ""; set myport ""; set keepalive ""
    set host ""; set port ""

    set i 0; set n [llength $args]
    while {$i < $n} {
        set a [lindex $args $i]
        if {[string match -* $a]} {
            switch -- $a {
                -async {
                    set async 1
                    if {$i+1 < $n} {
                        set nxt [lindex $args [expr {$i+1}]]
                        if {[string is integer -strict $nxt]} { set async [expr {$nxt ? 1 : 0}]; incr i }
                    }
                }
                -myaddr { if {$i+1 < $n} { incr i; set myaddr [lindex $args $i] } }
                -myport { if {$i+1 < $n} { incr i; set myport [lindex $args $i] } }
                -keepalive {
                    set keepalive 1
                    if {$i+1 < $n} {
                        set nxt [lindex $args [expr {$i+1}]]
                        if {[string is integer -strict $nxt]} { set keepalive $nxt; incr i }
                    }
                }
                default {}
            }
        } elseif {$host eq ""} {
            set host $a
        } elseif {$port eq ""} {
            set port $a
        }
        incr i
    }
    if {$host eq "" || $port eq ""} { error "_tls_socket: args invalides: $args" }

    # socket TCP
    set sockopts {}
    if {$myaddr ne ""}   { lappend sockopts -myaddr $myaddr }
    if {$myport ne ""}   { lappend sockopts -myport $myport }
    if {$keepalive ne ""} { lappend sockopts -keepalive $keepalive }
    set s [::socket {*}$sockopts $host $port]
    if {$async} { catch { fconfigure $s -blocking 0 } } else { catch { fconfigure $s -blocking 1 } }
    catch { fconfigure $s -translation binary -encoding binary -buffering none }

    # Upgrade TLS via tls::import (avec SNI si possible)
    set imported 0
    foreach attempt {
        {-autoservername 1 -servername __HOST__}
        {-servername __HOST__}
        {}
    } {
        set cmd [list ::tls::import $s]
        if {[llength $attempt]} {
            foreach {k v} $attempt {
                if {$v eq "__HOST__"} { set v $host }
                lappend cmd $k $v
            }
        }
        if {![catch { {*}$cmd }]} { set imported 1; break }
    }
    if {!$imported} { catch {close $s}; error "tls::import failed for $host:$port" }
    return $s
}

if {$::dtc(have_tls)} {
    catch { ::http::register httpsdtc 443 ::dtc::_tls_socket }
}

# =========================
#        Config
# =========================
set ::dtc(timeout)             15000
set ::dtc(max_redirects)       5
set ::dtc(max_lines)           0
set ::dtc(debug)               2
set ::dtc(line_cooldown_ms)    600
set ::dtc(http_cooldown_ms)    1500
# Couleurs IRC : blanc sur gris sombre (comme !vdm)
set ::dtc(irc_fg)              0   ;# 0 = blanc
set ::dtc(irc_bg)              14  ;# 14 = gris sombre
set ::dtc(preview_lines)       10
set ::dtc(max_search_results)  5

# =========================
#      Chanset & state
# =========================
setudef flag DansTonChat
namespace eval ::dtc {
    variable last_send_ts; array set last_send_ts {}
    variable cookie_jar;   array set cookie_jar {}
    variable last_http_ts 0
}

# =========================
#       Logging helpers
# =========================
proc ::dtc::log0 {tag msg} { if {$::dtc(debug) >= 0} { putlog "DTC($tag): $msg" } }
proc ::dtc::log1 {tag msg} { if {$::dtc(debug) >= 1} { putlog "DTC($tag): $msg" } }
proc ::dtc::log2 {tag msg} { if {$::dtc(debug) >= 2} { putlog "DTC-2($tag): $msg" } }
proc ::dtc::info {msg} { ::dtc::log1 INFO $msg }
proc ::dtc::err  {msg} { ::dtc::log0 ERR  $msg }
proc ::dtc::http {msg} { ::dtc::log2 HTTP $msg }
proc ::dtc::pars {msg} { ::dtc::log2 PARSE $msg }

# =========================
#       Utilitaires
# =========================
proc ::dtc::rate_send {chan line} {
    set now [clock milliseconds]
    if {![::info exists ::dtc::last_send_ts($chan)]} { set ::dtc::last_send_ts($chan) 0 }
    set delta [expr {$now - $::dtc::last_send_ts($chan)}]
    if {$delta < $::dtc(line_cooldown_ms)} { after [expr {$::dtc(line_cooldown_ms) - $delta}] }
    puthelp "PRIVMSG $chan :$line"
    set ::dtc::last_send_ts($chan) [clock milliseconds]
}

proc ::dtc::urldecode {s} {
    set out ""; set i 0; set n [string length $s]
    while {$i < $n} {
        set c [string index $s $i]
        if {$c eq "%"} {
            if {$i+2 < $n} {
                set hex [string range $s [expr {$i+1}] [expr {$i+2}]]
                if {[regexp {^[0-9A-Fa-f]{2}$} $hex]} {
                    scan $hex %x code; append out [format %c $code]; incr i 3; continue
                }
            }
        } elseif {$c eq "+"} { append out " "; incr i; continue }
        append out $c; incr i
    }
    return $out
}

# Cookies
proc ::dtc::cookie_host {url} {
    if {[regexp {^https?://([^/]+)} $url -> h]} { return [string tolower $h] }
    if {[regexp {^httpsdtc://([^/]+)} $url -> h2]} { return [string tolower $h2] }
    return ""
}
proc ::dtc::cookie_build {host} {
    if {$host eq ""} { return "" }
    if {[::info exists ::dtc::cookie_jar($host)]} { return $::dtc::cookie_jar($host) }
    return ""
}
proc ::dtc::cookie_learn {host metaList} {
    if {$host eq ""} { return }
    for {set i 0} {$i < [llength $metaList]} {incr i 2} {
        set k [string tolower [lindex $metaList $i]]
        set v [lindex $metaList [expr {$i+1}]]
        if {$k eq "set-cookie"} {
            set nv [lindex [split $v ";"] 0]
            if {$nv ne ""} {
                set cur [::dtc::cookie_build $host]
                if {$cur eq ""} { set ::dtc::cookie_jar($host) $nv } \
                elseif {[string first $nv $cur] < 0} { set ::dtc::cookie_jar($host) "$cur; $nv" }
                ::dtc::http "cookie learned for $host => $::dtc::cookie_jar($host)"
            }
        }
    }
}

# =========================
#         HTTP GET
# =========================
proc ::dtc::_rewrite_url {url} {
    if {$::dtc(have_tls) && $::dtc(use_wrapper_https)} {
        if {[string match -nocase "https://*" $url]} {
            return [string map {"https://" "httpsdtc://"} $url]
        }
    }
    return $url
}

# Fetch HTTP (cooldown + headers + backoff 429)
proc ::dtc::fetch {url {depth 0}} {
    if {$depth > $::dtc(max_redirects)} {
        ::dtc::err "Max redirects on $url"
        return [list 599 "" {}]
    }
    # Cooldown global
    set now [clock milliseconds]
    set delta [expr {$now - $::dtc::last_http_ts}]
    if {$delta < $::dtc(http_cooldown_ms)} {
        set wait [expr {$::dtc(http_cooldown_ms) - $delta}]
        ::dtc::http "cooldown ${wait}ms"
        after $wait
    }

    set eff [::dtc::_rewrite_url $url]
    set host [::dtc::cookie_host $eff]
    set cookie [::dtc::cookie_build $host]
    set hdr [list \
        "Accept" "text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8" \
        "Connection" "close" \
        "Cache-Control" "no-cache" \
        "Pragma" "no-cache" \
        "Referer" "https://danstonchat.com/" \
        "Accept-Language" "fr-FR,fr;q=0.9,en;q=0.8" \
    ]
    if {$cookie ne ""} { lappend hdr "Cookie" $cookie }

    set attempt 0
    set max_attempts 4
    set last_code 0
    set last_body ""
    set last_meta {}

    while {$attempt < $max_attempts} {
        incr attempt
        ::dtc::http "GET $eff (try=$attempt, depth=$depth)"
        set tkn ""
        if {[catch {
            set tkn [::http::geturl $eff -timeout $::dtc(timeout) -headers $hdr]
        } err]} {
            ::dtc::err "HTTP error: $err"
            set last_code 598
            break
        }

        set code [::http::ncode $tkn]
        set meta [::http::meta  $tkn]
        set body ""; if {[::http::status $tkn] eq "ok"} { set body [::http::data $tkn] }
        ::http::cleanup $tkn

        ::dtc::cookie_learn $host $meta
        set ::dtc::last_http_ts [clock milliseconds]
        ::dtc::http "HTTP $code, bytes=[string length $body]"

        # Redirection ?
        set loc ""
        for {set i 0} {$i < [llength $meta]} {incr i 2} {
            if {[string tolower [lindex $meta $i]] eq "location"} { set loc [lindex $meta [expr {$i+1}]]; break }
        }
        if {$code >= 300 && $code < 400 && $loc ne ""} {
            if {![regexp {^https?://|^httpsdtc://} $loc]} {
                if {[regexp {^(https?://[^/]+)} $url -> origin]} {
                    if {[string index $loc 0] ne "/"} { set loc "/$loc" }
                    set loc "$origin$loc"
                }
            }
            ::dtc::http "redirect -> $loc"
            return [::dtc::fetch $loc [expr {$depth+1}]]
        }

        if {$code == 429 && $attempt < $max_attempts} {
            set retry_after ""
            for {set i 0} {$i < [llength $meta]} {incr i 2} {
                if {[string tolower [lindex $meta $i]] eq "retry-after"} { set retry_after [lindex $meta [expr {$i+1}]]; break }
            }
            set base [expr {($retry_after ne "" && [string is integer -strict $retry_after]) ? ($retry_after*1000)
                            : ($attempt==1 ? 1000 : $attempt==2 ? 3000 : $attempt==3 ? 7000 : 15000)}]
            set jitter [expr {int(rand()*300)}]
            set waitms [expr {$base + $jitter}]
            ::dtc::http "429 → retry in ${waitms}ms"
            after $waitms
            continue
        }

        set last_code $code; set last_body $body; set last_meta $meta
        break
    }
    return [list $last_code $last_body $last_meta]
}

# -------------------------
#   HTML helpers
# -------------------------
proc ::dtc::html_named_unescape {s} {
    set map [list \
        "&amp;" "&" "&lt;" "<" "&gt;" ">" "&quot;" "\"" "&apos;" "'" \
        "&nbsp;" " " "&thinsp;" " " "&ensp;" " " "&emsp;" " " \
        "&laquo;" "«" "&raquo;" "»" "&hellip;" "…" "&ndash;" "–" "&mdash;" "—" \
        "&lsquo;" "‘" "&rsquo;" "’" "&ldquo;" "“" "&rdquo;" "”" \
        "&oelig;" "œ" "&OElig;" "Œ" "&euro;" "€" \
    ]
    return [string map $map $s]
}
proc ::dtc::decode_numeric_entities {s} {
    while {[regexp -indices {&#([0-9]+);} $s m n]} {
        foreach {m0 m1} $m {}; foreach {n0 n1} $n {}
        set num [string range $s $n0 $n1]
        if {[catch {set ch [format %c $num]}]} { set ch "?" }
        set s [string replace $s $m0 $m1 $ch]
    }
    while {[regexp -indices {&#[xX]([0-9A-Fa-f]+);} $s m n]} {
        foreach {m0 m1} $m {}; foreach {n0 n1} $n {}
        set hex [string range $s $n0 $n1]
        if {[catch {set ch [format %c 0x$hex]}]} { set ch "?" }
        set s [string replace $s $m0 $m1 $ch]
    }
    return $s
}
proc ::dtc::html_unescape {s} {
    ::dtc::pars "unescape before len=[string length $s]"
    set s [::dtc::html_named_unescape $s]
    set s [::dtc::decode_numeric_entities $s]
    ::dtc::pars "unescape after  len=[string length $s]"
    return $s
}

# -------------------------
#   Extraction quote principale
# -------------------------
proc ::dtc::extract_first_quote {html} {
    set out ""
    if {[catch {
        set html [string map {"\r\n" "\n" "\r" "\n"} $html]
        ::dtc::pars "find <div class=entry-content>"
        set openIdx ""
        if {![regexp -nocase -indices {<div[^>]*class="[^"]*entry-content[^"]*"[^>]*>} $html openIdx]} {
            ::dtc::info "entry-content introuvable"
            return ""
        }
        foreach {openS openE} $openIdx {}
        set gt [string first ">" $html $openS]
        if {$gt < 0} { return "" }
        set contentStart [expr {$gt + 1}]

        set contentEnd -1
        if {[regexp -indices -nocase -start $contentStart {</div>\s*</div>} $html closePairIdx]} {
            foreach {cS cE} $closePairIdx {} ; set contentEnd [expr {$cS - 1}]
        } else {
            set pos $contentStart; set depth 1
            while {$depth > 0} {
                set nextOpen  [string first "<div"   $html $pos]
                set nextClose [string first "</div>" $html $pos]
                if {$nextClose < 0 && $nextOpen < 0} { return "" }
                if {$nextOpen >= 0 && ($nextOpen < $nextClose || $nextClose < 0)} { set pos [expr {$nextOpen + 4}]; incr depth } \
                else { set contentEnd [expr {$nextClose - 1}]; set pos [expr {$nextClose + 6}]; incr depth -1 }
                if {$pos >= [string length $html]} { break }
            }
        }
        if {$contentEnd < $contentStart} { return "" }

        set block [string range $html $contentStart $contentEnd]
        ::dtc::pars "block size=[string length $block]"
        regsub -all -nocase {<(br|/p|/li)\s*[^>]*>} $block "\n" block
        regsub -all -nocase {<p[^>]*>}             $block ""  block
        regsub -all -nocase {<li[^>]*>}            $block "• " block
        regsub -all {<[^>]*>}                      $block ""  block
        set block [::dtc::html_unescape $block]

        set lines [split $block "\n"]
        set kept {}
        foreach L $lines {
            set L [string trim $L]
            if {$L eq ""} { continue }
            if {[string trim $L] eq "•"} { continue }
            if {[regexp -nocase {^(sur cette page|par |le [0-9]|[0-9]{4}$)} $L]} { continue }
            if {[regexp -nocase {^(partager|cliquez pour partager|élément copié)} $L]} { continue }
            lappend kept $L
        }
        if {[llength $kept] == 0} { return "" }
        set out [join $kept \n]
    } err]} {
        ::dtc::err "extract error: $err"
        return ""
    }
    return $out
}

# -------------------------
#   SEARCH via DDG (+ pagination)
# -------------------------
proc ::dtc::url_search_ddg_narrow {q} {
    set query [format "site:danstonchat.com/quote %s" $q]
    set qs [::http::formatQuery q $query kl fr-fr kp -2 ia web]
    return "https://duckduckgo.com/html/?$qs"
}
proc ::dtc::url_search_ddg_broad {q} {
    set query [format "site:danstonchat.com %s" $q]
    set qs [::http::formatQuery q $query kl fr-fr kp -2 ia web]
    return "https://duckduckgo.com/html/?$qs"
}
proc ::dtc::url_search_ddg_ultralite {q} {
    set query [format "site:danstonchat.com %s" $q]
    set qs [::http::formatQuery q $query]
    return "https://lite.duckduckgo.com/lite/?$qs"
}

proc ::dtc::parse_ddg_ids {html} {
    set ids {}; array set seen {}
    set count_a 0; set count_l 0
    set start 0
    while {[regexp -indices -nocase -start $start {(?:https?://(?:www\.)?danstonchat\.com)?/quote/([0-9]+)\.html} $html m]} {
        foreach {s e} $m {}
        set frag [string range $html $s $e]
        if {[regexp {([0-9]+)} $frag -> id]} {
            if {![info exists seen($id)]} { set seen($id) 1; lappend ids $id; incr count_a }
        }
        set start [expr {$e + 1}]
    }
    set start 0
    while {[regexp -indices -nocase -start $start {href=['"][^"']*/l/\?[^"']*uddg=([^"'&]+)} $html m uIdx]} {
        foreach {s e} $m {}; foreach {us ue} $uIdx {}
        set enc [string range $html $us $ue]
        set dec [::dtc::urldecode $enc]
        if {[regexp -nocase {(?:https?://(?:www\.)?danstonchat\.com)?/quote/([0-9]+)\.html} $dec -> id]} {
            if {![info exists seen($id)]} { set seen($id) 1; lappend ids $id; incr count_l }
        }
        set start [expr {$e + 1}]
    }
    ::dtc::info "DDG parse: direct=$count_a, via-uddg=$count_l, total=[llength $ids]"
    if {$::dtc(debug) >= 2} {
        ::dtc::pars "DDG IDs: [join $ids , ]"
        if {[llength $ids] == 0} {
            set m [regexp -inline -all -nocase {/quote/[0-9]+\.html} $html]
            if {[llength $m] > 0} { ::dtc::pars "DDG has /quote hits but regex missed? sample=[lindex $m 0]" } \
            else { ::dtc::pars "DDG page contains no /quote/<id>.html" }
        }
    }
    return $ids
}

# -------------------------
#   Fallback WordPress REST API
# -------------------------
proc ::dtc::search_ids_wp {q} {
    # 1) /wp-json/wp/v2/search?search=...
    set qs [::http::formatQuery search $q per_page 20]
    set url "https://danstonchat.com/wp-json/wp/v2/search?$qs"
    ::dtc::info "SEARCH WP search: '$q'"
    lassign [::dtc::fetch $url] code body _
    ::dtc::info "WP search status=$code, bytes=[string length $body]"
    set ids [list]
    if {$code == 200 && $body ne ""} {
        set matches [regexp -inline -all -nocase {(?:https?://(?:www\.)?danstonchat\.com)?/quote/([0-9]+)\.html} $body]
        foreach m $matches { if {[regexp {([0-9]+)} $m -> id]} { if {[lsearch -exact $ids $id] < 0} { lappend ids $id } } }
        if {[llength $ids] > 0} { return $ids }
    }

    # 2) /wp-json/wp/v2/posts?search=...&_fields=link
    set qs2 [::http::formatQuery search $q per_page 20 _fields link]
    set url2 "https://danstonchat.com/wp-json/wp/v2/posts?$qs2"
    ::dtc::info "SEARCH WP posts: '$q'"
    lassign [::dtc::fetch $url2] code2 body2 _
    ::dtc::info "WP posts status=$code2, bytes=[string length $body2]"
    if {$code2 == 200 && $body2 ne ""} {
        set matches [regexp -inline -all -nocase {(?:https?://(?:www\.)?danstonchat\.com)?/quote/([0-9]+)\.html} $body2]
        foreach m $matches { if {[regexp {([0-9]+)} $m -> id]} { if {[lsearch -exact $ids $id] < 0} { lappend ids $id } } }
    }
    return $ids
}

# -------------------------
#   Recherche principale (DDG → WP → DTC)
# -------------------------
proc ::dtc::search_ids {q} {
    set ids {}

    # A) DDG (3 variantes × pagination)
    set variants [list \
        [list "DDG-narrow"    ::dtc::url_search_ddg_narrow    ::dtc::parse_ddg_ids] \
        [list "DDG-broad"     ::dtc::url_search_ddg_broad     ::dtc::parse_ddg_ids] \
        [list "DDG-ultralite" ::dtc::url_search_ddg_ultralite ::dtc::parse_ddg_ids] \
    ]
    foreach v $variants {
        lassign $v name urlf parsef
        set next [$urlf $q]
        set hops 0
        while {$hops < 3 && [llength $ids] == 0 && $next ne ""} {
            incr hops
            ::dtc::info "SEARCH $name hop#$hops: '$q'"
            lassign [::dtc::fetch $next] code html _
            ::dtc::info "$name status=$code, html-bytes=[string length $html]"
            if {$code == 200 && $html ne ""} {
                set more [$parsef $html]
                if {[llength $more] > 0} { set ids $more; break }
                # Page suivante ? (standard /html/ et /lite/)
                if {[regexp -nocase {href=['"](?:https?://(?:html\.|lite\.)?duckduckgo\.com)?(/html/\?q=[^"' ]+?&s=\d+[^"']*)['"]} $html -> rel]} {
                    set next "https://duckduckgo.com$rel"
                    ::dtc::pars "$name Next -> $next"
                    continue
                }
                if {[regexp -nocase {href=['"](?:https?://lite\.duckduckgo\.com)?(/lite/\?q=[^"' ]+?&s=\d+[^"']*)['"]} $html -> rel2]} {
                    set next "https://lite.duckduckgo.com$rel2"
                    ::dtc::pars "$name Next (lite) -> $next"
                    continue
                }
                ::dtc::pars "No $name Next link."
            }
            set next ""
        }
        if {[llength $ids] > 0} { ::dtc::info "$name yielded [llength $ids] id(s)"; break }
    }

    # B) WordPress REST API si DDG a rien donné
    if {[llength $ids] == 0} {
        set ids [::dtc::search_ids_wp $q]
        if {[llength $ids] > 0} {
            ::dtc::info "WP REST yielded [llength $ids] id(s)"
            return $ids
        }
    }

    # C) Fallback DTC (/?s=) en dernier recours
    if {[llength $ids] == 0} {
        set qs [::http::formatQuery s $q]
        set url2 "https://danstonchat.com/?$qs"
        ::dtc::info "SEARCH DTC (fallback): '$q'"
        lassign [::dtc::fetch $url2] code2 html2 _
        ::dtc::info "DTC fallback status=$code2, html-bytes=[string length $html2]"
        if {$code2 == 200 && $html2 ne ""} {
            set start 0
            while {[regexp -indices -nocase -start $start {href=['"](?:https?://(?:www\.)?danstonchat\.com)?/quote/([0-9]+)\.html['"]} $html2 m]} {
                foreach {s e} $m {}
                set frag [string range $html2 $s $e]
                if {[regexp {([0-9]+)} $frag -> id]} {
                    if {[lsearch -exact $ids $id] < 0} { lappend ids $id }
                }
                set start [expr {$e + 1}]
            }
        } else {
            ::dtc::info "DTC fallback: aucun résultat"
        }
    }
    return $ids
}

# -------------------------
#   URLs DTC
# -------------------------
proc ::dtc::url_random {} { return "https://danstonchat.com/random" }
proc ::dtc::url_for_id {id} { return "https://danstonchat.com/quote/${id}.html" }
proc ::dtc::url_for_id_amp {id} { return "https://danstonchat.com/quote/${id}.html/amp" }

# -------------------------
#   Fetchers
# -------------------------
proc ::dtc::fetch_random {} {
    ::dtc::info "RANDOM: fetch"
    lassign [::dtc::fetch [::dtc::url_random]] code html _
    if {$code != 200 || $html eq ""} { return [list "" ""] }
    return [::dtc::pick_random_quote_with_id $html]
}

proc ::dtc::fetch_by_id {id} {
    ::dtc::info "BY-ID: $id"
    lassign [::dtc::fetch [::dtc::url_for_id $id]] code html _
    ::dtc::info "BY-ID std status=$code, bytes=[string length $html]"
    if {$code == 200 && $html ne ""} {
        set q [::dtc::extract_first_quote $html]
        if {$q ne ""} { return [list $id [::dtc::strip_trailing_numeric_debris $q]] }
    }
    lassign [::dtc::fetch [::dtc::url_for_id_amp $id]] codeA htmlA _
    ::dtc::info "BY-ID amp status=$codeA, bytes=[string length $htmlA]"
    if {$codeA == 200 && $htmlA ne ""} {
        set h [string map {"\r\n" "\n" "\r" "\n"} $htmlA]
        set q ""
        if {[regexp -indices -nocase {<div[^>]*class="[^"]*amp-wp-article-content[^"]*"[^>]*>} $h open]} {
            foreach {s e} $open {}
            set gt [string first ">" $h $s]
            if {$gt >= 0} {
                set cs [expr {$gt+1}]
                set ce [string first "</div>" $h $cs]
                if {$ce < 0} { set ce [expr {[string length $h]-1}] }
                set block [string range $h $cs $ce]
                regsub -all -nocase {<(br|/p|/li)\s*[^>]*>} $block "\n" block
                regsub -all -nocase {<p[^>]*>}             $block ""  block
                regsub -all -nocase {<li[^>]*>}            $block "• " block
                regsub -all {<[^>]*>}                      $block ""  block
                set q [::dtc::html_unescape [string trim $block]]
            }
        }
        if {$q ne ""} { return [list $id [::dtc::strip_trailing_numeric_debris $q]] }
    }
    if {$html ne ""} {
        set h [string map {"\r\n" "\n" "\r" "\n"} $html]
        if {[regexp -indices -nocase {<blockquote[^>]*>} $h bi]} {
            foreach {bs be} $bi {}
            set gt [string first ">" $h $bs]
            if {$gt >= 0} {
                set cs [expr {$gt+1}]
                set ce [string first "</blockquote>" $h $cs]
                if {$ce > $cs} {
                    set block [string range $h $cs $ce]
                    regsub -all -nocase {<(br|/p|/li)\s*[^>]*>} $block "\n" block
                    regsub -all {<[^>]*>} $block "" block
                    set q [::dtc::html_unescape [string trim $block]]
                    if {$q ne ""} { return [list $id [::dtc::strip_trailing_numeric_debris $q]] }
                }
            }
        }
    }
    return [list "" ""]
}

# -------------------------
#   Helpers divers
# -------------------------
proc ::dtc::pick_random_quote_with_id {html} {
    set html [string map {"\r\n" "\n" "\r" "\n"} $html]
    set blocks {}; set start 0
    while {[regexp -indices -nocase -start $start {<div[^>]*class="[^"]*entry-content[^"]*"[^>]*>} $html m]} {
        foreach {s e} $m {}
        set gt [string first ">" $html $s]; if {$gt < 0} { break }
        set contentStart [expr {$gt + 1}]
        set contentEnd -1
        if {[regexp -indices -nocase -start $contentStart {</div>\s*</div>} $html m2]} {
            foreach {cs ce} $m2 {} ; set contentEnd [expr {$cs - 1}]
        } else {
            set pos $contentStart; set depth 1
            while {$depth > 0} {
                set nextOpen  [string first "<div"   $html $pos]
                set nextClose [string first "</div>" $html $pos]
                if {$nextClose < 0 && $nextOpen < 0} { break }
                if {$nextOpen >= 0 && ($nextOpen < $nextClose || $nextClose < 0)} { set pos [expr {$nextOpen + 4}]; incr depth } \
                else { set contentEnd [expr {$nextClose - 1}]; set pos [expr {$nextClose + 6}]; incr depth -1 }
                if {$pos >= [string length $html]} { break }
            }
        }
        if {$contentEnd > $contentStart} { lappend blocks [list $contentStart $contentEnd] }
        set start [expr {$e + 1}]
    }
    if {[llength $blocks] == 0} { return [list "" ""] }

    set pick [lindex $blocks [expr {int(rand()*[llength $blocks])}]]
    foreach {cs ce} $pick {}
    set block [string range $html $cs $ce]

    set rid ""
    if {[regexp -nocase {/quote/([0-9]+)\.html} $block -> qid]} { set rid $qid } \
    elseif {[regexp -nocase {data-id\s*=\s*"([0-9]+)"} $block -> qid]} { set rid $qid }
    if {$rid eq ""} {
        set winS [expr {$cs - 1200}]; if {$winS < 0} { set winS 0 }
        set winE [expr {$ce + 1200}]; if {$winE >= [string length $html]} { set winE [expr {[string length $html]-1}] }
        set win  [string range $html $winS $winE]
        if {[regexp -nocase {/quote/([0-9]+)\.html} $win -> qid2]} { set rid $qid2 } \
        elseif {[regexp -nocase {data-id\s*=\s*"([0-9]+)"} $win -> qid2]} { set rid $qid2 } \
        elseif {[regexp -nocase {<article[^>]*id="post-([0-9]+)"} $win -> qid2]} { set rid $qid2 }
    }
    if {$rid eq ""} {
        if {[regexp -nocase {/quote/([0-9]+)\.html} $html -> qid3]} { set rid $qid3 }
    }

    set text [::dtc::extract_first_quote "<div class=\"entry-content\">$block</div>"]
    return [list $rid $text]
}

proc ::dtc::maybe_truncate {text} {
    if {$::dtc(max_lines) <= 0} { return $text }
    set L [split $text "\n"]
    if {[llength $L] <= $::dtc(max_lines)} { return $text }
    set head [lrange $L 0 [expr {$::dtc(max_lines)-1}]]
    return "[join $head \n]\n[…]"
}

proc ::dtc::strip_trailing_numeric_debris {text} {
    set lines [split $text "\n"]
    while {[llength $lines] > 0 && [string trim [lindex $lines end]] eq ""} {
        set lines [lreplace $lines end end]
    }
    if {[llength $lines] == 0} { return "" }
    set last [string trim [lindex $lines end]]
    if {[regexp {^[0-9]+$} $last]} {
        set lines [lreplace $lines end end]
        return [join $lines "\n"]
    }
    return $text
}

# --- Helpers d’affichage (style !vdm) ---
# 1) Première ligne : [ID en blanc et gras] + espace + 1re ligne en blanc (même bulle, fond gris)
proc ::dtc::format_first_line {rid first_line} {
    set bg $::dtc(irc_bg)
    return [format "\0030,%d\002\[%s\]\002\0030,%d %s\017" $bg $rid $bg $first_line]
}
# 2) Lignes suivantes (sans ID, toujours blanc sur gris)
proc ::dtc::format_rest_lines {lines} {
    set fg $::dtc(irc_fg)
    set bg $::dtc(irc_bg)
    set out {}
    foreach L $lines {
        set L [string trim $L]
        if {$L eq ""} { continue }
        lappend out [format "\003%s,%s%s\017" $fg $bg $L]
    }
    return $out
}
# 3) Envoi complet d’une quote avec ID en préfixe de la 1re ligne
proc ::dtc::send_quote {chan rid text} {
    set txt  [::dtc::maybe_truncate $text]
    set raw  [split $txt "\n"]
    # extraire la première ligne non vide
    set first ""
    set rest {}
    foreach L $raw {
        set t [string trim $L]
        if {$t eq ""} { continue }
        if {$first eq ""} { set first $t } else { lappend rest $t }
    }
    if {$first eq ""} {
        ::dtc::rate_send $chan [format "\0030,%d\002\[%s\]\002\017" $::dtc(irc_bg) $rid]
        return
    }
    ::dtc::rate_send $chan [::dtc::format_first_line $rid $first]
    foreach L [::dtc::format_rest_lines $rest] {
        ::dtc::rate_send $chan $L
    }
}

# =========================
#         Bindings
# =========================
bind pub - "!dtc"      ::dtc::cmd
bind pub - "!bashfr"   ::dtc::cmd

# IMPORTANT : dernier paramètre = texte brut (pas une liste)
proc ::dtc::cmd {nick uhost hand chan text} {
    ::dtc::info "PUB on $chan from $nick text='[string range $text 0 120]'"

    if {![channel get $chan DansTonChat]} {
        puthelp "PRIVMSG $chan :Activez +DansTonChat pour !dtc / !bashfr (.chanset $chan +DansTonChat)"
        return
    }

    # Nettoyage de l'argument (tolérant aux parenthèses autour d'un nombre)
    set arg [string trim $text]

    if {$arg eq ""} {
        # Aléatoire
        lassign [::dtc::fetch_random] rid textq
        if {$textq eq ""} { ::dtc::rate_send $chan "Désolé, aucune quote aléatoire trouvée pour le moment."; return }
        if {$rid eq ""} { set rid "??" }
        ::dtc::send_quote $chan $rid $textq
        return
    }

    # Si c'est un ID, accepter "2", "(2)", "  2  ", etc.
    if {[regexp {^\s*\(?\s*([0-9]+)\s*\)?\s*$} $arg -> num]} {
        lassign [::dtc::fetch_by_id $num] rid textq
        if {$textq eq ""} { ::dtc::rate_send $chan "Désolé, je n’ai pas trouvé la quote #$num."; return }
        if {$rid eq ""} { set rid $num }
        ::dtc::send_quote $chan $rid $textq
        return
    }

    # Recherche texte (DDG → WP REST → DTC) → ids → fetch 1re quote
    set q $arg
    set ids [::dtc::search_ids $q]
    if {[llength $ids] == 0} {
        ::dtc::rate_send $chan "Aucun résultat pour « $q »."
        return
    }

    set show_ids [lrange $ids 0 [expr {$::dtc(max_search_results)-1}]]
    set bold_ids {}
    foreach id $show_ids { lappend bold_ids [format "\002%s\002" $id] }
    set header [format "\[DansTonChat : %s\]" [join $bold_ids "|"]]
    ::dtc::rate_send $chan $header

    set first_rid [lindex $ids 0]
    ::dtc::info "SEARCH => ids=[llength $ids], chosen=$first_rid"
    lassign [::dtc::fetch_by_id $first_rid] _ first_text
    if {$first_text ne ""} {
        ::dtc::send_quote $chan $first_rid $first_text
    } else {
        ::dtc::rate_send $chan "Désolé, impossible d’afficher l’aperçu pour #$first_rid."
    }
}

# =========================
#     Fin de chargement
# =========================
if {$::dtc(have_tls)} {
    putlog "DTC: script chargé (v$::dtc(version)) — cmds: !dtc / !bashfr — chanset requis: +DansTonChat"
} else {
    putlog "DTC: script chargé (v$::dtc(version)) — TLS absent, on tentera quand même. Cmds: !dtc / !bashfr"
}

catch { putlog "Dans Ton Chat script chargé by Te\[u\]K" }
